use strict;
use warnings;
package Mixin::Linewise::Writers;

our $VERSION = '0.002';

use Carp ();
use IO::File;
use IO::String;

use Sub::Exporter -setup => {
  exports => { map {; "write_$_" => \"_mk_write_$_" } qw(file string) },
  groups  => {
    default => [ qw(write_file write_string) ],
    writers => [ qw(write_file write_string) ],
  },
};

=head1 NAME

Mixin::Linewise::Writers - get linewise writeers for strings and filenames

=head1 SYNOPSIS

  package Your::Pkg;
  use Mixin::Linewise::Writers -writers;

  sub write_handle {
    my ($self, $data, $handle) = @_;

    $handle->print("datum: $_\n") for @$data;
  }

Then:

  use Your::Pkg;

  Your::Pkg->write_file($data, $filename);

  Your::Pkg->write_string($data, $string);

  Your::Pkg->write_handle($data, $fh);

=head1 EXPORTS

C<write_file> and C<write_string> are exported by default.  Either can be
requested individually, or renamed.  They are generated by
L<Sub::Exporter|Sub::Exporter>, so consult its documentation for more
information.

Both can be generated with the option "method" which requests that a method
other than "write_handle" is called with the created IO::Handle.

=head2 write_file

  Your::Pkg->write_file($data, $filename);

This method will try to open a new file with the given name.  It will then call
C<write_handle> with that handle.

Any arguments after C<$filename> are passed along after to C<write_handle>.

=cut

sub _mk_write_file {
  my ($self, $name, $arg) = @_;
  my $method = defined $arg->{method} ? $arg->{method} : 'write_handle';

  sub {
    my ($invocant, $data, $filename) = splice @_, 0, 3;

    # Check the file
    Carp::croak "no filename specified"           unless $filename;
    Carp::croak "'$filename' is not a plain file" if -e $filename && ! -f _;

    # Write out the file
    my $handle = IO::File->new($filename, '>')
      or Carp::croak "couldn't write to file '$filename': $!";

    $invocant->write_handle($data, $handle, @_);
  }
}

=head2 write_string

  my $string = Your::Pkg->write_string($data);

C<write_string> will create a new IO::String handle, call C<write_handle> to
write to that handle, and return the resulting string.

Any arguments after C<$data> are passed along after to C<write_handle>.

=cut

sub _mk_write_string {
  my ($self, $name, $arg) = @_;
  my $method = defined $arg->{method} ? $arg->{method} : 'write_handle';

  sub {
    my ($invocant, $data) = splice @_, 0, 2;

    my $string = '';
    my $handle = IO::String->new($string);

    $invocant->write_handle($data, $handle, @_);

    return $string;
  }
}

=head1 BUGS

Bugs should be reported via the CPAN bug tracker at

L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Mixin-Linewise>

For other issues, or commercial enhancement or support, contact the author.

=head1 AUTHOR

Ricardo SIGNES, C<< E<lt>rjbs@cpan.orgE<gt> >>

=head1 COPYRIGHT

Copyright 2008, Ricardo SIGNES.

This program is free software; you may redistribute it and/or modify it under
the same terms as Perl itself.

=cut

1;
