use strict;
use warnings;

use Test::More;
use Crypt::Mac::OMAC;
use Crypt::Cipher;

my $trans = {
  "3des"       => 'DES_EDE',
  "safer+"     => 'SAFERP',
  "khazad"     => 'Khazad',
  "safer-k128" => 'SAFER_K128',
  "safer-sk128"=> 'SAFER_SK128',
  "rc6"        => 'RC6',
  "safer-k64"  => 'SAFER_K64',
  "safer-sk64" => 'SAFER_SK64',
  "anubis"     => 'Anubis',
  "blowfish"   => 'Blowfish',
  "xtea"       => 'XTEA',
  "aes"        => 'AES',
  "rc5"        => 'RC5',
  "cast5"      => 'CAST5',
  "skipjack"   => 'Skipjack',
  "twofish"    => 'Twofish',
  "noekeon"    => 'Noekeon',
  "rc2"        => 'RC2',
  "des"        => 'DES',
  "camellia"   => 'Camellia',
};
my $tv;
my $name;
my $ks;

while (my $l = <DATA>) {
  $l =~ s/[\r\n]*$//;  
  $l =~ s/^[\s]*([^\s\r\n]+).*?/$1/;
  $l =~ s/\s+//;
  if ($l=~/^OMAC-([a-z0-9\+\-]+).*?(\d+)/i) {
    $name = $1;
    $ks = $2;
    next;
  }
  my ($k, $v) = split /:/, $l;
  next unless defined $k && defined $v;
  $tv->{$name}->{$ks}->{$k} = $v if $name && $k =~ /\d+/;
}

my $seq;
$seq .= pack('C',$_) for(0..255);
my $zeros = '\0' x 255;

for my $n (sort keys %$tv) {
  for my $ks (sort keys %{$tv->{$n}}) {
    my $N = $trans->{$n} || die "FATAL: unknown name '$n'";
    my $key = substr($seq, 0, $ks);  
    for my $i (0..255) {
      my $bytes = substr($seq, 0, $i);
      next unless $tv->{$n}->{$ks}->{$i};
      my $result = Crypt::Mac::OMAC->new($N, $key)->add($bytes)->mac;
      is(unpack('H*', $result), lc($tv->{$n}->{$ks}->{$i}), "$N/$i");
      $bytes = $result; 
      $key = substr($result x 100, 0, $ks);
    }
  }
}

done_testing();

__DATA__
OMAC Tests.  In these tests messages of N bytes long (00,01,02,...,NN-1) are OMAC'ed.  The initial key is
of the same format (length specified per cipher).  The OMAC key in step N+1 is the OMAC output of
step N (repeated as required to fill the array).

OMAC-aes (16 byte key)
  0: 97DD6E5A882CBD564C39AE7D1C5A31AA
  1: F69346EEB9A76553172FC20E9DB18C63
  2: 996B17202E2EDEBD63F414DD5E84F3AF
  3: D00D7DA967A2873589A7496503B3DBAB
  4: B43C24C0A82DAA12D328395C2ABD7CAE
  5: 9B902B6663B5FEDC6F9DCE74B35B91F2
  6: 06A9678C65D7CE225E082ECA31788335
  7: 7D67866CDB313DF65DED113DB02D6362
  8: 259E28CF3E578AC47A21A77BA9EA8261
  9: 32F23C8F93EA301C6D3FE0840CA8DB4B
 10: C2B06388AD6F8C43D19FE4F6A8ED21AE
 11: FA8622485DB2F62F84FF46E532A1A141
 12: F312D9B2E6272578F406B66C79F30A0E
 13: 7A5DE06B2BFB75ADA665E96F680AC098
 14: C3B00380F0BD8E2F5C9DD9945E0F36EE
 15: DDD87974A5FB2E7A4514241E94526B5B
 16: AD24FC47A0FEA84C54696DE997A94F4B
 17: 7538713D8AA2AE3726307EFF087BBF5E
 18: 7619A52B4C34A98440812F5F28F8DC4F
 19: 7E797B8846554888622CC5E400B2FA44
 20: 61E8DD3E09145F5657DB4B8F7BD2D7D8
 21: FDAE2A3FE60DDF1871C2613A293AB6F1
 22: A186D6EFD10DFFD2C088480B0A784185
 23: 3119D337865618CDA55C06FB992427CF
 24: 413E3EAD7E3F169A37C49F9CA92E235E
 25: 37A55AF22373B9A1E2F8368B2FB992CA
 26: 4941F604C40EEEE1A16CFE073C12D1FE
 27: 3E8F4A0876BF12A2DCA87157F15DC884
 28: 5DFAE292D8EEB13D8FE5725E5D169742
 29: 59160455E0C0B35D950BA67C77F9FB05
 30: 5AC0D736A06A7DD146B137ADEE78EE06
 31: 0CA1178F28B953045EE76E2E760036CA
 32: 025616215F870D1EF838AD1D2AE0C649

OMAC-blowfish (8 byte key)
  0: 2CFB5DE451FFE8CC
  1: A5AC339DB44D020C
  2: A3CE0CF62249444D
  3: 3076B7129CE3F6A1
  4: 9E091A637DDF70E3
  5: 275199AB20A5F09C
  6: CDEDA8D16A401E62
  7: FC980516CF5C9E30
  8: 659D0B31D21B622B
  9: 8306847B5E72E018
 10: 7AD029BBF1D2919F
 11: 133181425C6808C9
 12: FC5AC60E367F413A
 13: E0DF8BCCF0AD01D9
 14: AC5015398FA64A85
 15: 1F068F22AFFECEE1
 16: 8E6831D5370678EF

OMAC-xtea (16 byte key)
  0: A821403929958A1A
  1: 68C4A02D47C2E7C0
  2: 7D37358141506DC1
  3: 9BEA3AAE55B75F52
  4: 884D053D05CC8DE4
  5: E953747483FF4E0D
  6: B6E77E72C9738E4F
  7: 8AB67D2B24E3D512
  8: 329C0B9D504A0D41
  9: 50323DA8ACEF4164
 10: FA3239C668C34DA3
 11: B5A12FC81FC24084
 12: 71A01A3ED3936934
 13: F29B630CEB6AEDDB
 14: F8802F0D4504D55E
 15: F844B92162038F99
 16: 99AECD7CA69F0465

OMAC-rc5 (8 byte key)
  0: E374E40562C3CB23
  1: B46D83F69233E236
  2: 7CB72B1D335F04B0
  3: 94457CBC97B31328
  4: 543D0EDFCDCD7C76
  5: 5164EFA8412EAA5D
  6: 13CA0717EF95F9A7
  7: 2AA49A7AA7719700
  8: C9E7C56125C3D90F
  9: 2BE3E15FE58648AA
 10: 77D0B90372D6D0FD
 11: 17408F62ECD62F57
 12: 7864EFFA59DC059B
 13: 3212E76E25E5DEA8
 14: E2424C083CDE5A6A
 15: DE86FFDBDA65D138
 16: 85482C24D61B8950

OMAC-rc6 (16 byte key)
  0: E103BD8BA47B7C1C010E1561712E6722
  1: E51AEECFED3AF40443B3A1C011407736
  2: FA6506C5ABE03381B045D28D1D828966
  3: FAC4237FFE7772E2299D3D983BB130DD
  4: 3A7E24D41121A5D4F96FCECF0C2A4A10
  5: AA44291E5500C1C8E1A14CB56E4F979A
  6: 4B8FDA6DA6B3266E39111F403C31754E
  7: 4DF5F1A1C8EBC7F56D0D12EEB63FF585
  8: 46A6DDE419355EDE14D31045FCA1BA35
  9: 71756D4D3DF59578B7F93FD4B5C08187
 10: ADA292A19F8636A03A8BC58C26D65B0D
 11: 703190DAF17F8D08A67A11FDF0C2A622
 12: D2B94CAD1AFC5CD012575964D1425BE6
 13: 45FD0069FCA6F72E23E4DB41AA543091
 14: 36F652600F5C9F226721400A7199E2BA
 15: E8CC6389ECF8EF1DBB90A0FD051B7570
 16: 8125446B975DBDA742A903340D6B96C7
 17: 00B55E4399EB930E592F507F896BF3DC
 18: 33E58F42A47C9543A851D6CA9324FEE0
 19: 9F28FDEA3EC7F515128F5D0C0EB684C5
 20: AC1DAF6C01AA28BCC0A819189FA949D7
 21: D0532B5F54A179444D052A4D2AD6E4F9
 22: 58B80A66549404C7B9F64D5AE3F798AB
 23: D0D6D586477F92311DDF667E0749D338
 24: 0DFC0FAA67FF114398CE94D0688AE146
 25: E163B8C00CF5CC9FA23ACACD62B53D64
 26: ACE9270456AF9BD388BA72E98825CFE8
 27: 4302EED9BAA19C7A296585E23A066A44
 28: B3EEABEFAB25C7478419265564715387
 29: 9F0630ADE9C74AB2981D63F3B69E85BF
 30: 1215A9446A275CCE2714F94F3C213BB7
 31: AF43D7F748DE0E3458DB970BAC37E98D
 32: BF871AC9E892CE0DCD7C8C7ADDD854C6

OMAC-safer+ (16 byte key)
  0: A2C8C7FEA5529D01C3FF4E9359EF74F4
  1: EAB87021118FF24FE79B69ABCCB14A8F
  2: 789566F467BAA68F4CC3C4B61901D6D4
  3: 369F41EEAF7D628F9E0D77BE43BFC1D2
  4: DC46A20E1F36F45006ED5B43BEC20DA6
  5: 8F150CE34F57BBA2E6CE3431B78E4ACD
  6: 61CD154478BE20F33B26CD8FC58091A5
  7: 4E6DAA575CF28F1F48B256262B7D558C
  8: D21FA4F1859571DB91E92767C5487AA2
  9: E3D009DC7E71FBBB030B8FF0B544A2C9
 10: 094C236EA48ABF7DBAE5A88AA3DE07D7
 11: 00C401996F8224359566660AC1CEDAA1
 12: D580EC60F712558D875F01643D96653F
 13: 8482298027C7B4D5969787A1DB1B1F2F
 14: AB726AE3DA95CB242E63EF876A4BC446
 15: D668ED4919003F5E45590663FAED41DA
 16: E4CFFD7E0E7B176867C386001849FD6F
 17: 37B3C6DEFC5573879006D15F982A397C
 18: 0AB8847EE6A41A0E960080EF0D1BF1C5
 19: 2C94FCA2A685F276A65ED286AE12FD9F
 20: 23383032032D7B5165A31ECA156DBD23
 21: E1EECFB3D671DF694FFB05AE4305AD4C
 22: A0F6CA99B96CD1EDD04C52828C8A4D74
 23: 12D6B7053417AF3E407EFD6EE1CC38FE
 24: A566D1C39AE7A1A0A77D5A1F56C5FAAB
 25: 81C9FAECEAEA326140AFCD569668F669
 26: 6A00BF1D0DC893868378E4347CB4A1B9
 27: 98842956DBE7AFB1BF49C46497BD54C7
 28: 88EFCD5A1644B75BB0B3F5DD338849CE
 29: 77EC62C278C61163B1BEC595A11F047A
 30: 147424E817DC69413CC657E0CB292F7F
 31: A2946CBB910743EF62D8A3C7391B9B9B
 32: 00EEDA55520B8A5B88B76487E80EB6E1

OMAC-twofish (16 byte key)
  0: 0158EB365FCCFDD94EBA6BE42B6659C4
  1: 17DA580917D147D10CB73DB6800B0E59
  2: 3F185CC15EF3328D3E075665308C07C8
  3: 5712A97ACC9D08FE9D2087D0CA16B0AD
  4: 90425A8CC1C026DDD896FC2131AF654B
  5: 30A43D4FEAE71F5396308C16DA081B4A
  6: 6839FEF605704D49F1A379A9E9595E6F
  7: 56A8F06DFEE543971B351B07430E2026
  8: 36DD0E4B55C5314F9F2753D7EB6F0849
  9: 8E319249A3CD456460F410F518F8CEDB
 10: 463978BE2A063C22E71DC71520723517
 11: 1B735E45FD3DF636E0A6104D4A2E9CB8
 12: 628A82213148AD9791153D5AAFBDDFDC
 13: 21AFDF08A36ADB6659B656C8EA0800E5
 14: E5C3E58803DDBE174E0D4C2B8171AEF0
 15: FC6981F2B4359BA05988D61822C0FA88
 16: 7B03498FAFB04A6542248852225F9DAE
 17: 9B173E91E59A940186E57BB867B8307B
 18: 470BF2EE614C8423AA3FDF323F1C103E
 19: 6E664AFDFD8306547BBEDA036D267B79
 20: F61AEC1144C3DD646169E16073700AC6
 21: AE503B139707AFA494F7F2DE933EE81A
 22: A0A8BDD4ED0DCAE4A8E1DCEE56368FF0
 23: 460B8207930DA434AE6AFECC305D9A26
 24: 7F03F8C7BA5365CC65F7864A42693BC8
 25: 31448849D6190484192F29A221700011
 26: BDA941019C75551D858F70FB1362EB23
 27: 2880CB3E62447AE8EACA76C17971BB18
 28: FC8D710FA3990B56357E61C2A302EB84
 29: 793CD15348D7DFF301C47BC6E6235E22
 30: 6FB0CE69A15A3B6A933324A480077D35
 31: C24FCA5DD4AE0DF2BFF17364D17D6743
 32: DC6738080478AF9AF7CA833295031E06

OMAC-safer-k64 (8 byte key)
  0: 726FE2DD40A43924
  1: 2A138B65EB352621
  2: 9588A1B53E29616C
  3: C025DEFDE1A59850
  4: 73D062F1B6D8E003
  5: 944598A2FC8A2D76
  6: B176C25D8CAFFC98
  7: 14F05014DE6A090A
  8: A7B9847B2CE22D0F
  9: FCD71310CBAA3A62
 10: BFF00CE5D4A20331
 11: BEE12A2171333ED5
 12: 333FD849BEB4A64A
 13: D048EC7E93B90435
 14: F04960356689CFEF
 15: 9E63D9744BF1B61A
 16: 7C744982F32F8889

OMAC-safer-sk64 (8 byte key)
  0: E96711BA37D53743
  1: 7DCFF26A03509FE1
  2: 0A20EF19C8EE9BF2
  3: FE2883748A6963CF
  4: 557060195B820A18
  5: 771A7931FBBE5C0F
  6: 6BDBCE5F96CF91D8
  7: F3B924CCE8724595
  8: EC7191286D83C2C3
  9: 94F55B19BB7A8AC1
 10: 2189F4F2B06A8CA4
 11: 99853DAEBCA33A46
 12: 66EAC37A033802D7
 13: 845D7AA866F8A8AD
 14: 33A874DFECAC22AC
 15: 63DD9F7A7F3683DF
 16: EAC277D951676C44

OMAC-safer-k128 (16 byte key)
  0: 8037B89AF193F129
  1: FF2314E87BA6AFE1
  2: C3243DF896B61D85
  3: 0F61C715CE821AB8
  4: EBFDC6A9CFD2F5A4
  5: AB6497D7AF2C7FFF
  6: C920CEEB7C1819C2
  7: 3E186951B545A7E5
  8: 5EA36A93C94AF4AC
  9: 6A2C59FAE33709BE
 10: BF1BAFAF9FC39C19
 11: 69EB6EF046677B7C
 12: CDDCEE6B20453094
 13: A3833BD3FED6895C
 14: B6C05E51F01E049B
 15: 90A2D0EAB739D39B
 16: 07BF607A161D0A66

OMAC-safer-sk128 (16 byte key)
  0: 5E8B137A3946A557
  1: 0228FA66B13F3C7E
  2: A6F9BBAFF050DCDD
  3: F75880F684A796CE
  4: E0AEFB8E32040EBD
  5: 9F65D658B86D310F
  6: 3FA52804FB46CCAA
  7: 2F6D12D199FCD2FB
  8: CB56AF60AFB4D2BB
  9: 8E6F0FF6FDD262FD
 10: 490245BE3CCCEDE2
 11: EFD319AE46C73005
 12: 43E00E545C848995
 13: 10444B41ECA15EBE
 14: 521775C389D5BE71
 15: 9B683EF8B097FEBA
 16: 3C5D746EED09530A

OMAC-rc2 (8 byte key)
  0: F001FE9BBC3A97B0
  1: 8F8DC9C952897FBD
  2: EC82EAD195AAC38C
  3: 53DD52269B19E9A4
  4: 9B86F64BF72A0647
  5: 664A88A29F2898C6
  6: AFEC3F71C1415666
  7: 9BA1F2C1A2E765F9
  8: 402A12120908B436
  9: 03ECCD4C6AF44144
 10: E8CA3529B5D9D6FC
 11: 951EE10779CC585D
 12: B9083CA88E7E819B
 13: AFFB9E884DACC5B7
 14: E942E8BC241343D6
 15: 9B190489091344FB
 16: 9330A9E05554A15A

OMAC-des (8 byte key)
  0: C9085E99D74DF01D
  1: FAC84F0EFBEF8630
  2: C37C5FECE671CF16
  3: 45B2CBEE8701A5B1
  4: 53665E1F024EB001
  5: 357123CEDFC9FF61
  6: BD2CFD33FB1F832B
  7: 1AAA9D8C9120BDBF
  8: EB9F589AE9D4E78F
  9: C8F9D2ACE691922D
 10: 81ED6F3611DDC0FD
 11: 2965ABEAC46839EE
 12: 2208B1E095F7AE2E
 13: C0414FE41800113E
 14: 653A24119CF43D97
 15: 7FB7CE0862958B37
 16: 55097816B10C549B

OMAC-3des (24 byte key)
  0: 7F07A9EA8ECEDF9E
  1: 4E2A652EB5FBF5F8
  2: 4F84E3779ACCB9F5
  3: 7134AB3463115DC6
  4: 82327BE8EA2D7E0B
  5: 24950B9C14D87CD9
  6: B25A097BB7E0E18A
  7: ED51BAE55ED925E7
  8: 56B79E7644556975
  9: A65BD98E4D4E31E2
 10: 11145BB51514482D
 11: 397486787E676BA6
 12: BD1F6DEBAF6D9AEF
 13: 5CC3921F7DB815CF
 14: B0C0E60DA5F727F3
 15: F8637AEEFF10F470
 16: 0EA19531D42706EA

OMAC-cast5 (8 byte key)
  0: 7413DCDB9F0C3100
  1: 423799EDF1472B79
  2: 03856F0CB4F11606
  3: F152AE6360813DE0
  4: 853998BD980AD146
  5: AE6C3D667DB8B414
  6: B5A4986A34BDE20F
  7: E5ABE5B979798942
  8: BEE8DFED4555F405
  9: 6B5339E952AF61BE
 10: 5E867CF34D9C1149
 11: F9C55CB3BC655E08
 12: EA09A2929AC7D915
 13: CE8EB0E4370E1933
 14: 749A424B2AA91B98
 15: 8DDA93C2B814D5D1
 16: E8B0B219D4CB699B

OMAC-noekeon (16 byte key)
  0: A9F50A6567FAD338AB5727B3B94DEB82
  1: C9EC17EF3656C9056E64E692A449AD24
  2: 7D1F6A3373BF20D4E65804B745D40855
  3: FE3484F11C338721F3FCB4DCC608BD6E
  4: C45670D31D48CE9AD70BADE9F7A6A5B3
  5: 6AF86480F1AE638DCAC40939B864D4DE
  6: CBBAFED3A5891A7BD8692055E4C59444
  7: B23439FC6D1CF0E3B04BE5201CAF9283
  8: 385D2C64F55B3FE976E660155FAC4C90
  9: 239D4B8F663248076E64CF743AC14EC0
 10: B942C5E06C6E68866440EB10747643B6
 11: 9B591FA2FD9A20C367FB03366150D1E7
 12: F90183F872D062AB4642DCDCED399970
 13: 86003C2F260EAFC81BC45A0614F88381
 14: C80F88A148FF906D42E6D75A50049468
 15: 0A81478238ED815D7CB04C0DC5A4A4D5
 16: DFE74730DB9CF4F994084C88923A8931
 17: 91194DAAAB458B5B34E991EF534D4BD8
 18: DAA1CCA0B644AB9F8B4D889D7F1268FB
 19: A93AE4F41DFB6CA311FAAA148A9D53D9
 20: 6FFD47B80A991A6C09775060E4A4B13E
 21: 4BE3101511BCA3251559ED6D3BFCC304
 22: 3C1AA3485241175A9A17F440A1F2FF5F
 23: 11D2C5E4FC75639CC61C8FE66C2F5135
 24: 3EDBF9F32259650ABC2C835301FA6556
 25: 56FA9AA43C01CA5BA798780D3BF40FA1
 26: 40AE1F352003026C6D4C5F44430DD5ED
 27: 264E11C88266029588A1B8369F0C5B73
 28: 60CE0E6D2C2C74D122DBDE57B3EA44AB
 29: 5E4078E7CEFA94886E9CF0D083C4B468
 30: CEC169560600ECEED6E5C8F06C76E702
 31: B4209736F08EAAE6D5B4923D83EB3EE2
 32: 5DC8D45C9954B82864F1C2388858D97B

OMAC-skipjack (10 byte key)
  0: 84EDFA769040603C
  1: 7DA58A4CBD642627
  2: 118F60115CFC8229
  3: A7F7346D34DB2F0E
  4: 35615CCD526CD57F
  5: DE471601A3660844
  6: 15FCCE6D6D883D1F
  7: C6F694861233151B
  8: 3B762B397F16E807
  9: 976C6AB59FB3AB12
 10: 6810791F2C595961
 11: 7FA3478286917F17
 12: 73DEE44A51C6B610
 13: 89EE8B253B1ACE81
 14: CDF2586A56C8A0B5
 15: ED91F98DA98F42C4
 16: D8D0FA5CE96B08BF

OMAC-anubis (16 byte key)
  0: E672617CAA1E641C0E7B4B4CC4787455
  1: C0C16E8FD63907C08A8ABBB7B73376D3
  2: 23F97CED54939361830396224A7BDD91
  3: 7FD87DEA9F05E07212DDF61292D9E13D
  4: 929A11A4D0991A6446B1051926A6048D
  5: 4EB74F1CC0150D86126BC6FE1FC8253D
  6: 33C2C3C072D05BB6D54F87579C23B116
  7: DE350181C9E90A79879813A609BE77E2
  8: DB519EB9EF0E154D9D248734FD3D3724
  9: 4F7F2E6D3FC72BA94FE24EC0ABBF4E66
 10: D646389DBCEEDD59EBB6E8F09C422930
 11: 8547658AE1CE6A8B8D010A1E1FEA7AF4
 12: C9BE2B7F3630EFDFBD3AEA6A108C86EA
 13: 290417C57096B8B9A1BA3C20FD91285B
 14: 9AF60E99692C5F911CBF969A6E11DC14
 15: CDA433BE58C98E49EBA8A7108E50DE2B
 16: 7430D0EE631A4659351B8A4489A78D46
 17: DCC74C0FD0415768FE00225CA14B7DC2
 18: 0CF2432B1B465F2A8C5FACAAF2FEF619
 19: DA020680C64E93AE5FCA3D71466D01C1
 20: B9C33A86E6ED9FCCDCD973382DD1B6A3
 21: 6631236B9F2F810DD4D97E6046F41AF2
 22: 0312C322F4D634CF4FBC0C2624E3E9F2
 23: 111E3E9F8FBDC1E4364622723F1CB524
 24: 6D2608D7AAF243D5219E14513895BFF6
 25: 683BD01B43CBC0430A007ACBAB357DC9
 26: 01B8FC65C56B0F1A5BFEBEDCCF6748D9
 27: 4D6298D63A80D55491697A6DD8E3694C
 28: 6F0205E4E083CAB00747D723300510DF
 29: 5183BAEEF05E9402A935EB9AFF0AA2A9
 30: 1E673BFAD4944643A740C59D96A5925C
 31: 940FB4000E34EEE78E8DB402E4A76502
 32: 87B0C48F3D155AD85D0502D94A4572DE

OMAC-khazad (16 byte key)
  0: 4EBEFA460499424F
  1: 97AEEAD51E541D16
  2: 29A35212910C9595
  3: ABD1577D622074EA
  4: 70A537DE14DD765C
  5: 240A19016DE99C51
  6: 4D42C10A9F803177
  7: F464BC3E0DB5A909
  8: 1C65A01A7C08DAC7
  9: E49A1428C230C209
 10: 16DD0FEB7A6505B8
 11: 2DDDB3E35A05C220
 12: EC88910C799AC6CC
 13: B2A65C9EF39BEC8A
 14: F0D2366BA91DFFD5
 15: BCAB623CAB7AAA23
 16: 9BCEAB857596E478

OMAC-camellia (16 byte key)
  0: B5664C5148FFB45297703BCC46C19E4E
  1: 126EC31A554E8B3B635DE4617092ECE8
  2: FD57148685F4AA85AF48017AFD72B410
  3: 1427607464A408C1775B4036509E9659
  4: D8F5A7112CC8A9DF28B331FE7184BF08
  5: 0E29B0F09409DABECF645F05C4A5717C
  6: C4155442FDC29685028AF4AADEDCC266
  7: 92356ACB98AE2EDAABE0D3ED0C90772B
  8: AA3C828618F72258D91BC391876776C3
  9: 189458BA4D98E85802E7028E5C57A25F
 10: EE652D70328DA00D63B42A5E85D70E63
 11: F9D1E5F8E1539F2D657A047755CD232E
 12: 56FF5979FD3DEAD90EAAAF79A9AF1DCD
 13: 7E8B39D459D5AB449A8C5917B0CD0C4E
 14: 822D9B9C434C6FF7F0E5A25281740A91
 15: 654909D2836CCB06501CB359C717C1B9
 16: E8996FC89D47C91543B7BA3DC1C34B73
 17: DC29D51B2372DD7564CF56AF8702924F
 18: AD74D081197644DFE2723CABC991B1AC
 19: 26145C6DF074CA53125F6F386FBEA373
 20: 72C6C760A70FE410FAD113D8BE711D75
 21: 099D3B5802D7FB699B6B8F031BE10B3F
 22: A9D5DD3988A18AA7BC6F9C050BDBE8D2
 23: F7E99E4C3C7D127FF04FF325F7B06997
 24: E99A2F7547B5C6EDF3BC2EC2B8F05198
 25: 46C42FF49FCCFC49FBC99FEB08FEF10A
 26: DC349D600A754F73ACE6A7D2D00D3551
 27: FC2E5434ABBA44ABD9D724A9BB6CA2A6
 28: BA923927BF0074AD73BA8A6914194297
 29: 7DAB39F8D7E5CB93265568E6713C7CCD
 30: 9F60259B759B68E1C8F89CC36C7E170E
 31: 7D611F8BFEF0491CED8815C0E3D4CAFF
 32: 31E04DE5F9D1403C660E39891DE0D8DE
