############################################
# Rule is a base class for a chain of
# responsibility pattern for SLOC counting
# and hopefully a building spot for more
# complex SLOC counting.
# See POD for more details
############################################
package Utility::Rule;

use strict;
use warnings;

sub new{
	my($class) = @_;
	my $struct = 
	{
		PASS => 1,
		FAIL => 0,
		STOP => -1,
	};
	return bless $struct, $class;
}

sub appy{
	my($self) = @_;
	die "Rule::Apply is abstract";
}

package Utility::Rule::CommentRule;

use base ('Utility::Rule');

sub apply{
	my ($self, $line) = @_;
	return $self->{FAIL} if $line =~ /^#/;
	return $self->{PASS};	
}

package Utility::Rule::BracketRule;

use base ('Utility::Rule');

sub apply{
	my ($self, $line) = @_;
	return $self->{FAIL} if $line eq '}';
	return $self->{PASS};	
}

package Utility::Rule::WhitespaceRule;

use base ('Utility::Rule');

sub apply{
	my ($self, $line) = @_;
	return $self->{FAIL} unless $line;
	return $self->{PASS};	
}

package Utility::Rule::EndAndDataRule;

use base ('Utility::Rule');

sub apply{
	my ($self, $line) = @_;
	return $self->{STOP} if (($line eq '__END__') || ($line eq '__DATA__'));
	return $self->{PASS};	
}

1;

__END__

=pod

=head1 NAME

Rule.pm

=head1 SUMMARY

Rule is a base class for a pattern called Chain of Responsibility.
Each rule determines if the line meets one of three conditions.
It test whether it passed, failed, or if the counter should be stopped.

=head2 Rule

=head3 new

Creation method.  
No parameters.
There are three properties that should be used as return values.

$self->{PASS}
$self->{FAIL}
$self->{STOP}

=head3 apply

Takes a string, and determines if the string counts.
If the string passes the rules test, return PASS, else return FAIL.
If the rule determines that the counter should stop, return STOP.
It may also do more complicated stuff internally if needed.

=head2 WhitespaceRule

Returns pass unless the line is whitespace.

=head2 EndAndDataRule

Returns pass unless it finds and '__END__' or '__DATA__' section.
If it does find either one, it returns a stop.

=head2 BracketRule

Returns pass unless the line is a '}'.

=head2 CommentRule

Returns pass unless the line begins with a '#'.

=head1 AUTHOR

johnfraser <johnfraser@somewhere.com<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (c) 2005, John Fraser C<< <raptnor@yahoo.com> >>. All rights reserved.

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself. See L<perlartistic>.


=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.
=cut