#!perl -T
use warnings; use strict;
use Test::More tests => 22;
use Test::Fatal;

use lib '.';
use t::Elive;

use Elive;
use Elive::Connection;
use Elive::Entity::Recording;
use Elive::Entity::Meeting;
use XML::Simple;

use version;

our $t = Test::More->builder;

my $class = 'Elive::Entity::Recording';

my @data;
$data[0] = 'the quick brown fox. %%(&)+(*)+*(_+. This is unplayable junk!';
$data[1] = join('',map {pack('C', $_)} (0..255));

########################################################################
##   ** Mock tests **
########################################################################

do {
    my %result = t::Elive->test_connection(only => 'mock');
    my $auth = $result{auth};

    my $connection_class = $result{class};
    my $mock_connection = $connection_class->connect(@$auth);
    Elive->connection($mock_connection);
    my $now = time();

    my $meeting = Elive::Entity::Meeting->insert({
	name => 'created by t/soap-recording.t',
	facilitatorId => Elive->login->userId,
	start => $now . '000',
	end => ($now+900) . '000',
	privateMeeting => 1,
    });
    isa_ok($meeting, 'Elive::Entity::Meeting');

    my $recording = Elive::Entity::Recording->insert(
    {
	facilitator => Elive->login->userId,
	meetingId => $meeting->meetingId,
	data => $data[0],
	version => Elive->server_details->version,
	roomName => 'room t/soap-recording.t',
	creationDate => $now.'000',
	open => 1,
    },
    );

    isa_ok($recording, 'Elive::Entity::Recording');
    is($recording->size, length($data[0]), 'recording size as expected');

    Elive->disconnect;
};

########################################################################
##   ** Live tests **
########################################################################

SKIP: {

    my %result = t::Elive->test_connection(only => 'real');
    my $auth = $result{auth};

    skip ($result{reason} || 'skipping live tests', 19)
	unless $auth;

    my $connection_class = $result{class};
    my $connection = $connection_class->connect(@$auth);
    Elive->connection($connection);
    
    my $server_details =  Elive->server_details
	or skip ("unable to get server details - are all services running?", 19);

    my $recording_id = time()."000_$$".sprintf("%04d", rand(9999));
    my $room_name = 'generated by soap-recording.t (unplayable!)';

    note "uploading recording: $recording_id";
    my $recording;

    is( exception {
	$recording = Elive::Entity::Recording->upload(
	    {
		recordingId => $recording_id,
		roomName => $room_name,
		version => $server_details->version,
		data => $data[0],
		open => 0,
	    },
	    )
	      } => undef,
	      "upload recording - lives",
	);

    is( exception {
	$recording->update({roomName => $room_name.' (updated)', open => 1})
	      } => undef,
	      'Recording update - lives');

    isa_ok($recording, $class, 'uploaded recording');
    is($recording->recordingId, $recording_id, 'uploaded recording id as expected');
    is($recording->roomName, $room_name.' (updated)','uploaded recording name as expected');

    is($recording->size, length($data[0]), 'recording size as expected');

    ok($recording->open, 'recording open status as expected'); 

    my $recordings = Elive::Entity::Recording->list;

    ok($recordings && @$recordings, 'got list of recordings');

    ok(do{grep {$_->recordingId eq $recording_id} @$recordings},
       'uploaded recording found in recordings');

    my $version = version->parse($server_details->version)->numify;
    if ($version <= '9.005000') {
	$t->skip("recording downloads - not supported for Elluminate Live! <= 9.50 ($version)")
	    for 1..3;
    }
    else {

	my $data_download = $recording->download;

	ok($data_download, 'got recording download');
	ok(length($data_download) == length($data[0]),
	   sprintf('download has expected size %d bytes', length($data[0])),
	    );

	is($data_download, $data[0], 'downloaded data matches upload');
    }

    my $recordingJNLP;

    is( exception {
	$recordingJNLP = $recording->buildJNLP(
	    userId => Elive->login->userId,
	    userIP => '192.168.0.1',
	    )} => undef,
	     "buildJNLP - lives",
	);

    ok($recordingJNLP && !ref($recordingJNLP), 'got recording JNLP');
    is( exception {XMLin($recordingJNLP)} => undef, 'JNLP is valid XML');
    ok(my $web_url = $recording->web_url, 'got recording web_url()');
		
    $recording = undef;

    ok($recording = Elive::Entity::Recording->retrieve($recording_id), 'recording retrieval');

    is( exception {$recording->delete} => undef, 'recording delete - lives');

    isnt( exception {Elive::Entity::Recording->retrieve($recording_id)} => undef, "retrieval after delete as expected");

}

Elive->disconnect;

