#! /usr/bin/perl -w
# Test the bzip2 compressed log file

# Copyright (c) 2005-2007 imacat
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

use 5.005;
use strict;
use warnings;
use diagnostics;
use Test;

BEGIN { plan tests => 16 }

use File::Basename qw(basename);
use File::Spec::Functions qw(catdir catfile updir);
use FindBin;
use lib $FindBin::Bin;
use _helper;
use vars qw($te $WORKDIR $reslog $nofile $nobzip2);
use vars qw($fs1 $ft1 $fr1 $fs2 $fr2);

$te = "application/x-bzip2";
$WORKDIR = catdir($FindBin::Bin, "logs");
$reslog = catfile($FindBin::Bin, updir, "blib", "script", "reslog");

# If we have the file type checker somewhere
$nofile =   eval { require File::MMagic; 1; }
            || defined whereis "file"?
    undef: "File::MMagic or file executable not available";
# If we have bzip2 support somewhere
$nobzip2 =  eval { require Compress::Bzip2; import Compress::Bzip2 2.00; 1; }
            || defined whereis "bzip2"?
    undef: "Compress::Bzip2 v2 or bzip2 executable not available";

$fs1 = catfile($WORKDIR, "access_log.bz2");
$ft1 = catfile($WORKDIR, "access_log.tmp-reslog");
$fr1 = catfile($WORKDIR, "access_log.resolved.bz2");
$fs2 = catfile($WORKDIR, "access_log.ct.bz2");
$fr2 = catfile($WORKDIR, "access_log.rsd.bz2");

# 1: The default keep behavior
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cr1, $tr1);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs1;
    $reslog =~ s/"/\\"/g;
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr1)), flist $WORKDIR);
    ($cr1, $tr1) = (fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 2: Keep all
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs1;
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1 -k a), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fs1, $fr1)), flist $WORKDIR);
    ($cs1, $cr1, $tr1) = (fread $fs1, fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fs1: source incorrect.\nGot:\n$cs1\nExpected:\n$cs0\n"
        if $cs1 ne $cs0;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 3: Keep restart
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs1;
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1 -k r), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fs1, $fr1)), flist $WORKDIR);
    ($cs1, $cr1, $tr1) = (fread $fs1, fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fs1: source incorrect.\nGot:\n$cs1\nExpected \"\"\n"
        if $cs1 ne "";
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 4: Keep delete
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs1;
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1 -k d), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr1)), flist $WORKDIR);
    ($cr1, $tr1) = (fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 5: The default override behavior
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cr0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    ($cs0, $cr0) = (mkrndlog $fs1, mkrndlog $fr1);
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fs1, $fr1)), flist $WORKDIR);
    ($cs1, $cr1, $tr1) = (fread $fs1, fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno == 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fs1: source incorrect.\nGot:\n$cs1\nExpected:\n$cs0\n"
        if $cs1 ne $cs0;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cr0\n"
        if $cr1 ne $cr0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 6: Override overwrite
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cr0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    ($cs0, $cr0) = (mkrndlog $fs1, mkrndlog $fr1);
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1 -o o), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr1)), flist $WORKDIR);
    ($cr1, $tr1) = (fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 7: Override append
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cr0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    ($cs0, $cr0) = (mkrndlog $fs1, mkrndlog $fr1);
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1 -o a), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr1)), flist $WORKDIR);
    ($cr1, $tr1) = (fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cr0$cs0\n"
        if $cr1 ne $cr0 . $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 8: Override fail
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cr0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    ($cs0, $cr0) = (mkrndlog $fs1, mkrndlog $fr1);
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1 -o f), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fs1, $fr1)), flist $WORKDIR);
    ($cs1, $cr1, $tr1) = (fread $fs1, fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno == 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fs1: source incorrect.\nGot:\n$cs1\nExpected:\n$cs0\n"
        if $cs1 ne $cs0;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cr0\n"
        if $cr1 ne $cr0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 9: From file to STDOUT
$_ = eval {
    return if $nobzip2 || $nofile;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs1;
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1 -c), $fs1;
    frwrite $fr1, $out;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fs1, $fr1)), flist $WORKDIR);
    ($cs1, $cr1, $tr1) = (fread $fs1, fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fs1: source incorrect.\nGot:\n$cs1\nExpected:\n$cs0\n"
        if $cs1 ne $cs0;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2 || $nofile, $_, 1, $@);

# 10: From STDIN to STDOUT
$_ = eval {
    return if $nobzip2 || $nofile;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cs1, $cr1, $tr1);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs1;
    ($retno, $out, $err) = runcmd frread $fs1, $reslog, qw(-d -d -d -n 1);
    frwrite $fr1, $out;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fs1, $fr1)), flist $WORKDIR);
    ($cs1, $cr1, $tr1) = (fread $fs1, fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fs1: source incorrect.\nGot:\n$cs1\nExpected:\n$cs0\n"
        if $cs1 ne $cs0;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2 || $nofile, $_, 1, $@);

# 11: Attach a custom suffix
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cs1, $cr2, $tr2);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs1;
    ($retno, $out, $err) = runcmd $cs0, $reslog, qw(-d -d -d -n 1 -s .rsd), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr2)), flist $WORKDIR);
    ($cs1, $cr2, $tr2) = (fread $fs1, fread $fr2, ftype $fr2);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr2: result type incorrect.\nGot: $tr2\nExpected: $te\n"
        if !$nofile && $tr2 ne $te;
    die "$fr2: result incorrect.\nGot:\n$cr2\nExpected:\n$cs0\n"
        if $cr2 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 12: Trim the file name suffix
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cs2, $cr1, $tr1);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs2;
    ($retno, $out, $err) = runcmd $cs0, $reslog, qw(-d -d -d -n 1 -t .ct), $fs2;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr1)), flist $WORKDIR);
    ($cs2, $cr1, $tr1) = (fread $fs2, fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 13: Trim and attach our suffix
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cs2, $cr2, $tr2);
    mkcldir $WORKDIR;
    $cs0 = mkrndlog $fs2;
    ($retno, $out, $err) = runcmd $cs0, $reslog, qw(-d -d -d -n 1 -t .ct -s .rsd), $fs2;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr2)), flist $WORKDIR);
    ($cs2, $cr2, $tr2) = (fread $fs2, fread $fr2, ftype $fr2);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr2: result type incorrect.\nGot: $tr2\nExpected: $te\n"
        if !$nofile && $tr2 ne $te;
    die "$fr2: result incorrect.\nGot:\n$cr2\nExpected:\n$cs0\n"
        if $cr2 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 14: Stop for the temporary working file
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $ct0, $cs1, $ct1);
    mkcldir $WORKDIR;
    ($cs0, $ct0) = (mkrndlog $fs1, mkrndlog $ft1);
    ($retno, $out, $err) = runcmd $cs0, $reslog, qw(-d -d -d -n 1), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fs1, $ft1)), flist $WORKDIR);
    ($cs1, $ct1) = (fread $fs1, fread $ft1);
    rmalldir $WORKDIR;
    die $out . $err if $retno == 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fs1: source incorrect.\nGot:\n$cs1\nExpected:\n$cs0\n"
        if $cs1 ne $cs0;
    die "$ft1: temporary working file incorrect.\nGot:\n$ct1\nExpected:\n$ct0\n"
        if $ct1 ne $ct0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 15: No IP to be resolved
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cs0, $cr1, $tr1);
    mkcldir $WORKDIR;
    $cs0 = mknoiprndlog $fs1;
    $reslog =~ s/"/\\"/g;
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr1)), flist $WORKDIR);
    ($cr1, $tr1) = (fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected:\n$cs0\n"
        if $cr1 ne $cs0;
    1;
};
skip($nobzip2, $_, 1, $@);

# 16: Empty log file
$_ = eval {
    return if $nobzip2;
    my ($retno, $out, $err);
    my ($fl, $fl1, $cr1, $tr1);
    mkcldir $WORKDIR;
    fwrite $fs1, "";
    $reslog =~ s/"/\\"/g;
    ($retno, $out, $err) = runcmd "", $reslog, qw(-d -d -d -n 1), $fs1;
    ($fl, $fl1) = (join(" ", sort map basename($_), ($fr1)), flist $WORKDIR);
    ($cr1, $tr1) = (fread $fr1, ftype $fr1);
    rmalldir $WORKDIR;
    die $out . $err if $retno != 0;
    die "result files incorrect.\nGot: $fl1\nExpected: $fl\n"
        if $fl1 ne $fl;
    die "$fr1: result type incorrect.\nGot: $tr1\nExpected: $te\n"
        if !$nofile && $tr1 ne $te;
    die "$fr1: result incorrect.\nGot:\n$cr1\nExpected: Empty\n"
        if $cr1 ne "";
    1;
};
skip($nobzip2, $_, 1, $@);
