package Data::Timeline::SVK;

use strict;
use warnings;
use DateTime::Format::DateParse;


our $VERSION = '0.02';


use base qw(Data::Timeline::Builder);


__PACKAGE__->mk_scalar_accessors(qw(base_dir));


sub create {
    my $self = shift;

    my $timeline = $self->make_timeline;

    chdir $self->base_dir or
        die sprintf "can't chdir to %s: $!\n", $self->base_dir;

    open my $fh, '-|', "svk log" or die "can't read from pipe 'svk log': $!\n";
    while (<$fh>) {
        next unless /^(r\d+:\s*\w+)\s*\|\s*(20.*)/;
        $timeline->entries_push($self->make_entry(
            description => $1,
            timestamp   => DateTime::Format::DateParse->parse_datetime($2),
            type        => 'svk',
        ));
    }
    close $fh or die "can't close pipe to 'svk log': $!\n";
    $timeline;
}


1;


__END__



=head1 NAME

Data::Timeline::SVK - Builds a timeline from an 'svk log'


=head1 SYNOPSIS

    my $svk_timeline = Data::Timeline::SVK->new(
        base_dir => $svk_base_dir
    )->create;

=head1 DESCRIPTION

This class parses the output of C<svk log> in a given directory and creates a
timeline with entries corresponding to commits. Only the revision number and
the committer name are recorded in the entry descriptions.

The timeline entries generated by this builder have the type C<svk>.

This class manually parses the output of C<svk log>; it would probably be
better to use classes provided by the L<SVK> distribution, but because of its
dependency on L<SVN::Core>, the L<SVK> distribution is a bit difficult to
install.

Data::Timeline::SVK inherits from L<Data::Timeline::Builder>.

The superclass L<Data::Timeline::Builder> defines these methods and
functions:

    new(), make_entry(), make_timeline()

The superclass L<Class::Accessor::Complex> defines these methods and
functions:

    carp(), cluck(), croak(), flatten(), mk_abstract_accessors(),
    mk_array_accessors(), mk_boolean_accessors(),
    mk_class_array_accessors(), mk_class_hash_accessors(),
    mk_class_scalar_accessors(), mk_concat_accessors(),
    mk_forward_accessors(), mk_hash_accessors(), mk_integer_accessors(),
    mk_new(), mk_object_accessors(), mk_scalar_accessors(),
    mk_set_accessors(), mk_singleton()

The superclass L<Class::Accessor> defines these methods and functions:

    _carp(), _croak(), _mk_accessors(), accessor_name_for(),
    best_practice_accessor_name_for(), best_practice_mutator_name_for(),
    follow_best_practice(), get(), make_accessor(), make_ro_accessor(),
    make_wo_accessor(), mk_accessors(), mk_ro_accessors(),
    mk_wo_accessors(), mutator_name_for(), set()

The superclass L<Class::Accessor::Installer> defines these methods and
functions:

    install_accessor(), subname()

The superclass L<Class::Accessor::Constructor> defines these methods and
functions:

    NO_DIRTY(), WITH_DIRTY(), _make_constructor(), mk_constructor(),
    mk_constructor_with_dirty(), mk_singleton_constructor()

The superclass L<Data::Inherited> defines these methods and functions:

    every_hash(), every_list(), flush_every_cache_by_key()

The superclass L<Class::Accessor::Constructor::Base> defines these methods
and functions:

    HYGIENIC(), STORE(), clear_dirty(), clear_hygienic(),
    clear_unhygienic(), contains_hygienic(), contains_unhygienic(),
    delete_hygienic(), delete_unhygienic(), dirty(), dirty_clear(),
    dirty_set(), elements_hygienic(), elements_unhygienic(), hygienic(),
    hygienic_clear(), hygienic_contains(), hygienic_delete(),
    hygienic_elements(), hygienic_insert(), hygienic_is_empty(),
    hygienic_size(), insert_hygienic(), insert_unhygienic(),
    is_empty_hygienic(), is_empty_unhygienic(), set_dirty(),
    size_hygienic(), size_unhygienic(), unhygienic(), unhygienic_clear(),
    unhygienic_contains(), unhygienic_delete(), unhygienic_elements(),
    unhygienic_insert(), unhygienic_is_empty(), unhygienic_size()

The superclass L<Tie::StdHash> defines these methods and functions:

    CLEAR(), DELETE(), EXISTS(), FETCH(), FIRSTKEY(), NEXTKEY(), SCALAR(),
    TIEHASH()

=head1 METHODS

=over 4

=item base_dir

    my $value = $obj->base_dir;
    $obj->base_dir($value);

A basic getter/setter method. If called without an argument, it returns the
value. If called with a single argument, it sets the value.

=item base_dir_clear

    $obj->base_dir_clear;

Clears the value.

=item clear_base_dir

    $obj->clear_base_dir;

Clears the value.

=item base_dir

The directory from which the C<svk log> command is issued.

=item create

Start parsing the output of C<svk log> and return a L<Data::Timeline> object
with the commit information.

=back

=head1 TAGS

If you talk about this module in blogs, on del.icio.us or anywhere else,
please use the C<datatimelinesvk> tag.

=head1 VERSION 
                   
This document describes version 0.02 of L<Data::Timeline::SVK>.

=head1 BUGS AND LIMITATIONS

No bugs have been reported.

Please report any bugs or feature requests to
C<<bug-data-timeline-svk@rt.cpan.org>>, or through the web interface at
L<http://rt.cpan.org>.

=head1 INSTALLATION

See perlmodinstall for information and options on installing Perl modules.

=head1 AVAILABILITY

The latest version of this module is available from the Comprehensive Perl
Archive Network (CPAN). Visit <http://www.perl.com/CPAN/> to find a CPAN
site near you. Or see <http://www.perl.com/CPAN/authors/id/M/MA/MARCEL/>.

=head1 AUTHOR

Marcel GrE<uuml>nauer, C<< <marcel@cpan.org> >>

=head1 COPYRIGHT AND LICENSE

Copyright 2007 by Marcel GrE<uuml>nauer

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.


=cut

