use strict;
use warnings;
package Mixin::Linewise::Writers;
{
  $Mixin::Linewise::Writers::VERSION = '0.004';
}
# ABSTRACT: get linewise writeers for strings and filenames

use Carp ();
use IO::File;
use IO::String;

use Sub::Exporter -setup => {
  exports => { map {; "write_$_" => \"_mk_write_$_" } qw(file string) },
  groups  => {
    default => [ qw(write_file write_string) ],
    writers => [ qw(write_file write_string) ],
  },
};


sub _mk_write_file {
  my ($self, $name, $arg) = @_;
  my $method = defined $arg->{method} ? $arg->{method} : 'write_handle';

  sub {
    my ($invocant, $data, $filename) = splice @_, 0, 3;

    # Check the file
    Carp::croak "no filename specified"           unless $filename;
    Carp::croak "'$filename' is not a plain file" if -e $filename && ! -f _;

    # Write out the file
    my $handle = IO::File->new($filename, '>')
      or Carp::croak "couldn't write to file '$filename': $!";

    $invocant->write_handle($data, $handle, @_);
  }
}


sub _mk_write_string {
  my ($self, $name, $arg) = @_;
  my $method = defined $arg->{method} ? $arg->{method} : 'write_handle';

  sub {
    my ($invocant, $data) = splice @_, 0, 2;

    my $string = '';
    my $handle = IO::String->new($string);

    $invocant->write_handle($data, $handle, @_);

    return $string;
  }
}

1;

__END__

=pod

=head1 NAME

Mixin::Linewise::Writers - get linewise writeers for strings and filenames

=head1 VERSION

version 0.004

=head1 SYNOPSIS

  package Your::Pkg;
  use Mixin::Linewise::Writers -writers;

  sub write_handle {
    my ($self, $data, $handle) = @_;

    $handle->print("datum: $_\n") for @$data;
  }

Then:

  use Your::Pkg;

  Your::Pkg->write_file($data, $filename);

  Your::Pkg->write_string($data, $string);

  Your::Pkg->write_handle($data, $fh);

=head1 EXPORTS

C<write_file> and C<write_string> are exported by default.  Either can be
requested individually, or renamed.  They are generated by
L<Sub::Exporter|Sub::Exporter>, so consult its documentation for more
information.

Both can be generated with the option "method" which requests that a method
other than "write_handle" is called with the created IO::Handle.

=head2 write_file

  Your::Pkg->write_file($data, $filename);

This method will try to open a new file with the given name.  It will then call
C<write_handle> with that handle.

Any arguments after C<$filename> are passed along after to C<write_handle>.

=head2 write_string

  my $string = Your::Pkg->write_string($data);

C<write_string> will create a new IO::String handle, call C<write_handle> to
write to that handle, and return the resulting string.

Any arguments after C<$data> are passed along after to C<write_handle>.

=head1 AUTHOR

Ricardo SIGNES <rjbs@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2008 by Ricardo SIGNES.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
