package Color::Library::Dictionary::NBS_ISCC::TC;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::TC;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::TC - (NBS/ISCC TC) Standard Color Card of America

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (TC) Standard Color Card of America

    Textile Color Card Association [TCCA],
    Standard Color Card of America and U.S. Army Color Card 

The 237 names of this idiosyncratic dictionary map to 267 colors, but only 129 out of 267 possible centroids are called out.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#TC>

=head1 COLORS

	african brown       africanbrown     #3e322c

	african brown       africanbrown     #5b504f

	almond green        almondgreen      #5e716a

	amberlite           amberlite        #c19a6b

	american beauty     americanbeauty   #673147

	amethyst            amethyst         #86608e

	apple red           applered         #be0032

	apricot             apricot          #d99058

	aqua                aqua             #66ada4

	arbutus pink        arbutuspink      #dea5a4

	ashes of rose       ashesofrose      #915f6d

	autumn brown        autumnbrown      #6f4e37

	autumn brown        autumnbrown      #635147

	baby blue           babyblue         #bcd4e6

	baby pink           babypink         #f9ccca

	beaver              beaver           #635147

	beige               beige            #958070

	bisque              bisque           #958070

	black               black            #222222

	blue bird, bluebird bluebirdbluebird #436b95

	blue flower         blueflower       #91a3b0

	blue spruce         bluespruce       #3b7861

	bluesteel           bluesteel        #536878

	blue turquoise      blueturquoise    #66aabc

	bois de rose        boisderose       #905d5d

	bottle green        bottlegreen      #1b4d3e

	brick red           brickred         #722f37

	brittany blue       brittanyblue     #367588

	bronze              bronze           #6c541e

	brown,              brown            #6f4e37

	buff                buff             #d99058

	buff                buff             #c19a6b

	burgundy            burgundy         #673147

	burgundy            burgundy         #38152c

	burnished straw     burnishedstraw   #a67b5b

	burnt orange        burntorange      #cb6d51

	caramel brown       caramelbrown     #6f4e37

	caramel brown       caramelbrown     #422518

	cardinal            cardinal         #841b2d

	carmine             carmine          #be0032

	castor              castor           #5b504f

	catawba             catawba          #673147

	chalk pink          chalkpink        #dea5a4

	chamois             chamois          #c9ae5d

	champagne           champagne        #fad6a5

	chartreuse          chartreuse       #beb72e

	cherry              cherry           #b3446c

	claret              claret           #673147

	cobalt blue         cobaltblue       #00304e

	cocoa               cocoa            #79443b

	copenhagen          copenhagen       #536878

	coral               coral            #ea9399

	cork                cork             #a67b5b

	cornflower blue     cornflowerblue   #545aa7

	crab apple          crabapple        #d9603b

	crayon green        crayongreen      #93c592

	cream               cream            #f0ead6

	crimson             crimson          #78184a

	crocus              crocus           #b784a7

	dahlia purple       dahliapurple     #602f6b

	dark blue           darkblue         #222222

	dark cardinal       darkcardinal     #722f37

	duckling            duckling         #004958

	dustblu             dustblu          #91a3b0

	dusty pink          dustypink        #d9a6a9

	ecru                ecru             #ae9b82

	eggplant            eggplant         #291e29

	eggshell            eggshell         #f3e5ab

	electric            electric         #367588

	emerald             emerald          #44944a

	evergreen           evergreen        #355e3b

	evergreen           evergreen        #173620

	fawn                fawn             #7e6d5a

	flame red           flamered         #d9603b

	flax                flax             #c2b280

	flemish blue        flemishblue      #536878

	flesh pink          fleshpink        #ecd5c5

	forget-me-not       forgetmenot      #70a3cc

	fuchsia             fuchsia          #ce4676

	garnet              garnet           #722f37

	geranium            geranium         #be0032

	gold                gold             #ab9144

	gold brown          goldbrown        #ae6938

	gold brown          goldbrown        #80461b

	golden orange       goldenorange     #ed872d

	golden poppy        goldenpoppy      #e25822

	golden yellow       goldenyellow     #f3c300

	goldmist            goldmist         #c2b280

	graphite blue       graphiteblue     #36454f

	graphite blue       graphiteblue     #202830

	graphite blue       graphiteblue     #252440

	graphite blue       graphiteblue     #4c516d

	grebe               grebe            #555555

	grecian rose        grecianrose      #c48379

	green               green            #355e3b

	grotto blue         grottoblue       #007791

	gull                gull             #c1b6b3

	harvard crimson     harvardcrimson   #78184a

	henna               henna            #79443b

	homage blue         homageblue       #202830

	homage blue         homageblue       #252440

	honeydew            honeydew         #d99058

	hunter              hunter           #355e3b

	hydrangea blue      hydrangeablue    #436b95

	imperial purple     imperialpurple   #602f6b

	independence        independence     #252440

	independence        independence     #4c516d

	indian orange       indianorange     #e25822

	irish green         irishgreen       #00543d

	ivory               ivory            #f0ead6

	jade green          jadegreen        #679267

	jasmine             jasmine          #f8de7e

	jungle green        junglegreen      #004b49

	khaki               khaki            #967117

	lacquer             lacquer          #9e4732

	lavender            lavender         #aa98a9

	leghorn             leghorn          #f3e5ab

	lemon yellow        lemonyellow      #fada5e

	light blue          lightblue        #367588

	light blue          lightblue        #536878

	light olive drab    lightolivedrab   #6c541e

	lilac               lilac            #86608e

	limepeel            limepeel         #8a9a5b

	lupine              lupine           #8791bf

	lupine              lupine           #8c92ac

	lustre blue         lustreblue       #436b95

	magenta             magenta          #78184a

	magenta             magenta          #673147

	mahogany            mahogany         #79443b

	maize               maize            #f8de7e

	majolica blue       majolicablue     #436b95

	maple sugar         maplesugar       #826644

	marine corps        marinecorps      #00304e

	maroon              maroon           #722f37

	maroon              maroon           #673147

	maroon              maroon           #543d3f

	mauve               mauve            #875692

	melon pink          melonpink        #ffb7a5

	midnight            midnight         #202830

	mintleaf            mintleaf         #44944a

	mistiblu            mistiblu         #536878

	myrtle              myrtle           #1b4d3e

	national flag blue  nationalflagblue #252440

	natural             natural          #bfb8a5

	navy 1              navy1            #00304e

	navy 2              navy2            #00304e

	nickel              nickel           #8f817f

	nickel              nickel           #848482

	nile                nile             #c9dc89

	nile                nile             #8a9a5b

	nude                nude             #ae9b82

	nugget gold         nuggetgold       #af8d13

	oakwood             oakwood          #6f4e37

	old blue            oldblue          #91a3b0

	old china           oldchina         #436b95

	old glory blue      oldgloryblue     #4e5180

	old glory red       oldgloryred      #be0032

	old gold            oldgold          #ab9144

	old gold            oldgold          #c9ae5d

	old rose            oldrose          #c08081

	olive               olive            #4a5d23

	orange              orange           #ed872d

	orange              orange           #e25822

	orchid              orchid           #b695c0

	oriental blue       orientalblue     #0067a5

	pale blue           paleblue         #b2beb5

	pale blue           paleblue         #b9b8b5

	palmetto            palmetto         #5e716a

	pansy               pansy            #604e97

	pansy               pansy            #32174d

	paprica             paprica          #e25822

	parma violet        parmaviolet      #604e97

	parrot blue         parrotblue       #317873

	parrot blue         parrotblue       #367588

	pastel blue         pastelblue       #bcd4e6

	pastel pink         pastelpink       #dea5a4

	peach               peach            #d9a6a9

	peach blossom       peachblossom     #e4717a

	peach blossom       peachblossom     #a8516e

	peacock             peacock          #367588

	pearl grey          pearlgrey        #bfb8a5

	pearl grey          pearlgrey        #b9b8b5

	peasant blue        peasantblue      #536878

	peking blue         pekingblue       #436b95

	peking blue         pekingblue       #00304e

	periwinkle blue     periwinkleblue   #8791bf

	pigeon              pigeon           #8f817f

	pimento             pimento          #be0032

	plum                plum             #563c5c

	popcorn             popcorn          #c9ae5d

	primitive green     primitivegreen   #007959

	princeton orange    princetonorange  #ed872d

	prune               prune            #563c5c

	prune               prune            #301934

	purple              purple           #2f2140

	purple orchid       purpleorchid     #9e4f88

	putty               putty            #958070

	putty               putty            #8e8279

	raspberry           raspberry        #a8516e

	raspberry           raspberry        #673147

	redgrape            redgrape         #673147

	reseda              reseda           #5e716a

	reseda              reseda           #3a4b47

	river blue          riverblue        #367588

	river blue          riverblue        #536878

	robin's egg         robinsegg        #536878

	rose beige          rosebeige        #a67b5b

	rose pink           rosepink         #ea9399

	royal blue          royalblue        #00304e

	royal blue          royalblue        #00416a

	ruby                ruby             #78184a

	salmon              salmon           #cb6d51

	salmon pink         salmonpink       #f88379

	sand                sand             #bfb8a5

	sandalwood          sandalwood       #a67b5b

	sapphire blue       sapphireblue     #272458

	saxe blue           saxeblue         #536878

	scarab green        scarabgreen      #3eb489

	scarlet             scarlet          #be0032

	scarlet             scarlet          #bc3f4a

	seal                seal             #3e322c

	sea pink            seapink          #e4717a

	shell pink          shellpink        #f88379

	shrimp pink         shrimppink       #e4717a

	shrimp pink         shrimppink       #e66761

	silver              silver           #848482

	silver gray         silvergray       #848482

	sistine             sistine          #91a3b0

	sky blue            skyblue          #91a3b0

	sky blue            skyblue          #00304e

	smoke               smoke            #555555

	spanish yellow      spanishyellow    #eaa221

	spicebrown          spicebrown       #6f4e37

	spraygreen          spraygreen       #8da399

	spring green        springgreen      #7e9f2e

	starlight blue      starlightblue    #8c92ac

	steel               steel            #8b8589

	stone blue          stoneblue        #536878

	strawberry          strawberry       #a8516e

	sunset              sunset           #fad6a5

	tan                 tan              #c19a6b

	tangerine           tangerine        #d9603b

	tarragon            tarragon         #679267

	taupe               taupe            #5b504f

	teal                teal             #004958

	tea rose            tearose          #f88379

	terra cotta         terracotta       #9e4732

	terra cotta         terracotta       #882d17

	terra cotta         terracotta       #79443b

	tigerlily           tigerlily        #e66761

	tile blue           tileblue         #536878

	toast brown         toastbrown       #6f4e37

	tobacco             tobacco          #6f4e37

	topaz               topaz            #be8a3d

	tourmaline          tourmaline       #c7e6d7

	turquoise           turquoise        #66ada4

	ultramarine blue    ultramarineblue  #00a1c2

	ultramarine blue    ultramarineblue  #30267a

	vassar rose         vassarrose       #d597ae

	violet              violet           #604e97

	violine pink        violinepink      #a8516e

	west point          westpoint        #536878

	white               white            #f0ead6

	white               white            #f2f3f4

	white               white            #f0ead6

	white               white            #bfb8a5

	yale blue           yaleblue         #272458

	yellow              yellow           #eaa221


=cut

sub _load_color_list() {
    return [
['nbs-iscc-tc:africanbrown.62','africanbrown','african brown',[62,50,44],'3e322c',4076076],
['nbs-iscc-tc:africanbrown.64','africanbrown','african brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-tc:almondgreen.150','almondgreen','almond green',[94,113,106],'5e716a',6189418],
['nbs-iscc-tc:amberlite.76','amberlite','amberlite',[193,154,107],'c19a6b',12687979],
['nbs-iscc-tc:americanbeauty.259','americanbeauty','american beauty',[103,49,71],'673147',6762823],
['nbs-iscc-tc:amethyst.223','amethyst','amethyst',[134,96,142],'86608e',8806542],
['nbs-iscc-tc:applered.11','applered','apple red',[190,0,50],'be0032',12451890],
['nbs-iscc-tc:apricot.53','apricot','apricot',[217,144,88],'d99058',14258264],
['nbs-iscc-tc:aqua.163','aqua','aqua',[102,173,164],'66ada4',6729124],
['nbs-iscc-tc:arbutuspink.5','arbutuspink','arbutus pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-tc:ashesofrose.262','ashesofrose','ashes of rose',[145,95,109],'915f6d',9527149],
['nbs-iscc-tc:autumnbrown.58','autumnbrown','autumn brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-tc:autumnbrown.61','autumnbrown','autumn brown',[99,81,71],'635147',6508871],
['nbs-iscc-tc:babyblue.184','babyblue','baby blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-tc:babypink.4','babypink','baby pink',[249,204,202],'f9ccca',16370890],
['nbs-iscc-tc:beaver.61','beaver','beaver',[99,81,71],'635147',6508871],
['nbs-iscc-tc:beige.60','beige','beige',[149,128,112],'958070',9797744],
['nbs-iscc-tc:bisque.60','bisque','bisque',[149,128,112],'958070',9797744],
['nbs-iscc-tc:black.267','black','black',[34,34,34],'222222',2236962],
['nbs-iscc-tc:bluebirdbluebird.182','bluebirdbluebird','blue bird, bluebird',[67,107,149],'436b95',4418453],
['nbs-iscc-tc:blueflower.185','blueflower','blue flower',[145,163,176],'91a3b0',9544624],
['nbs-iscc-tc:bluespruce.145','bluespruce','blue spruce',[59,120,97],'3b7861',3897441],
['nbs-iscc-tc:bluesteel.186','bluesteel','bluesteel',[83,104,120],'536878',5466232],
['nbs-iscc-tc:blueturquoise.172','blueturquoise','blue turquoise',[102,170,188],'66aabc',6728380],
['nbs-iscc-tc:boisderose.19','boisderose','bois de rose ',[144,93,93],'905d5d',9461085],
['nbs-iscc-tc:bottlegreen.146','bottlegreen','bottle green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-tc:brickred.16','brickred','brick red',[114,47,55],'722f37',7483191],
['nbs-iscc-tc:brittanyblue.173','brittanyblue','brittany blue',[54,117,136],'367588',3569032],
['nbs-iscc-tc:bronze.95','bronze','bronze',[108,84,30],'6c541e',7099422],
['nbs-iscc-tc:brown.58','brown','brown,',[111,78,55],'6f4e37',7294519],
['nbs-iscc-tc:buff.53','buff','buff',[217,144,88],'d99058',14258264],
['nbs-iscc-tc:buff.76','buff','buff',[193,154,107],'c19a6b',12687979],
['nbs-iscc-tc:burgundy.259','burgundy','burgundy',[103,49,71],'673147',6762823],
['nbs-iscc-tc:burgundy.260','burgundy','burgundy',[56,21,44],'38152c',3675436],
['nbs-iscc-tc:burnishedstraw.57','burnishedstraw','burnished straw',[166,123,91],'a67b5b',10910555],
['nbs-iscc-tc:burntorange.37','burntorange','burnt orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-tc:caramelbrown.58','caramelbrown','caramel brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-tc:caramelbrown.59','caramelbrown','caramel brown',[66,37,24],'422518',4334872],
['nbs-iscc-tc:cardinal.13','cardinal','cardinal',[132,27,45],'841b2d',8657709],
['nbs-iscc-tc:carmine.11','carmine','carmine',[190,0,50],'be0032',12451890],
['nbs-iscc-tc:castor.64','castor','castor',[91,80,79],'5b504f',5984335],
['nbs-iscc-tc:catawba.259','catawba','catawba',[103,49,71],'673147',6762823],
['nbs-iscc-tc:chalkpink.5','chalkpink','chalk pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-tc:chamois.87','chamois','chamois',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-tc:champagne.73','champagne','champagne',[250,214,165],'fad6a5',16438949],
['nbs-iscc-tc:chartreuse.99','chartreuse','chartreuse',[190,183,46],'beb72e',12498734],
['nbs-iscc-tc:cherry.255','cherry','cherry',[179,68,108],'b3446c',11748460],
['nbs-iscc-tc:claret.259','claret','claret',[103,49,71],'673147',6762823],
['nbs-iscc-tc:cobaltblue.183','cobaltblue','cobalt blue',[0,48,78],'00304e',12366],
['nbs-iscc-tc:cocoa.43','cocoa','cocoa',[121,68,59],'79443b',7947323],
['nbs-iscc-tc:copenhagen.186','copenhagen','copenhagen',[83,104,120],'536878',5466232],
['nbs-iscc-tc:coral.2','coral','coral',[234,147,153],'ea9399',15373209],
['nbs-iscc-tc:cork.57','cork','cork',[166,123,91],'a67b5b',10910555],
['nbs-iscc-tc:cornflowerblue.196','cornflowerblue','cornflower blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-tc:crabapple.35','crabapple','crab apple',[217,96,59],'d9603b',14245947],
['nbs-iscc-tc:crayongreen.135','crayongreen','crayon green',[147,197,146],'93c592',9684370],
['nbs-iscc-tc:cream.92','cream','cream',[240,234,214],'f0ead6',15788758],
['nbs-iscc-tc:crimson.256','crimson','crimson',[120,24,74],'78184a',7870538],
['nbs-iscc-tc:crocus.240','crocus','crocus',[183,132,167],'b784a7',12027047],
['nbs-iscc-tc:dahliapurple.219','dahliapurple','dahlia purple',[96,47,107],'602f6b',6303595],
['nbs-iscc-tc:darkblue.267','darkblue','dark blue',[34,34,34],'222222',2236962],
['nbs-iscc-tc:darkcardinal.16','darkcardinal','dark cardinal',[114,47,55],'722f37',7483191],
['nbs-iscc-tc:duckling.174','duckling','duckling',[0,73,88],'004958',18776],
['nbs-iscc-tc:dustblu.185','dustblu','dustblu',[145,163,176],'91a3b0',9544624],
['nbs-iscc-tc:dustypink.29','dustypink','dusty pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-tc:ecru.79','ecru','ecru',[174,155,130],'ae9b82',11443074],
['nbs-iscc-tc:eggplant.230','eggplant','eggplant',[41,30,41],'291e29',2694697],
['nbs-iscc-tc:eggshell.89','eggshell','eggshell ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-tc:electric.173','electric','electric ',[54,117,136],'367588',3569032],
['nbs-iscc-tc:emerald.131','emerald','emerald',[68,148,74],'44944a',4494410],
['nbs-iscc-tc:evergreen.137','evergreen','evergreen',[53,94,59],'355e3b',3497531],
['nbs-iscc-tc:evergreen.138','evergreen','evergreen',[23,54,32],'173620',1521184],
['nbs-iscc-tc:fawn.80','fawn','fawn',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-tc:flamered.35','flamered','flame red',[217,96,59],'d9603b',14245947],
['nbs-iscc-tc:flax.90','flax','flax',[194,178,128],'c2b280',12759680],
['nbs-iscc-tc:flemishblue.186','flemishblue','flemish blue',[83,104,120],'536878',5466232],
['nbs-iscc-tc:fleshpink.31','fleshpink','flesh pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-tc:forgetmenot.181','forgetmenot','forget-me-not',[112,163,204],'70a3cc',7381964],
['nbs-iscc-tc:fuchsia.254','fuchsia','fuchsia',[206,70,118],'ce4676',13518454],
['nbs-iscc-tc:garnet.16','garnet','garnet',[114,47,55],'722f37',7483191],
['nbs-iscc-tc:geranium.11','geranium','geranium',[190,0,50],'be0032',12451890],
['nbs-iscc-tc:gold.88','gold','gold',[171,145,68],'ab9144',11243844],
['nbs-iscc-tc:goldbrown.54','goldbrown','gold brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-tc:goldbrown.55','goldbrown','gold brown',[128,70,27],'80461b',8406555],
['nbs-iscc-tc:goldenorange.50','goldenorange','golden orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-tc:goldenpoppy.34','goldenpoppy','golden poppy',[226,88,34],'e25822',14833698],
['nbs-iscc-tc:goldenyellow.82','goldenyellow','golden yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-tc:goldmist.90','goldmist','goldmist',[194,178,128],'c2b280',12759680],
['nbs-iscc-tc:graphiteblue.187','graphiteblue','graphite blue',[54,69,79],'36454f',3556687],
['nbs-iscc-tc:graphiteblue.188','graphiteblue','graphite blue',[32,40,48],'202830',2107440],
['nbs-iscc-tc:graphiteblue.201','graphiteblue','graphite blue',[37,36,64],'252440',2434112],
['nbs-iscc-tc:graphiteblue.204','graphiteblue','graphite blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-tc:grebe.266','grebe','grebe',[85,85,85],'555555',5592405],
['nbs-iscc-tc:grecianrose.30','grecianrose','grecian rose',[196,131,121],'c48379',12878713],
['nbs-iscc-tc:green.137','green','green',[53,94,59],'355e3b',3497531],
['nbs-iscc-tc:grottoblue.169','grottoblue','grotto blue',[0,119,145],'007791',30609],
['nbs-iscc-tc:gull.10','gull','gull',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-tc:harvardcrimson.256','harvardcrimson','harvard crimson',[120,24,74],'78184a',7870538],
['nbs-iscc-tc:henna.43','henna','henna',[121,68,59],'79443b',7947323],
['nbs-iscc-tc:homageblue.188','homageblue','homage blue',[32,40,48],'202830',2107440],
['nbs-iscc-tc:homageblue.201','homageblue','homage blue',[37,36,64],'252440',2434112],
['nbs-iscc-tc:honeydew.53','honeydew','honeydew',[217,144,88],'d99058',14258264],
['nbs-iscc-tc:hunter.137','hunter','hunter',[53,94,59],'355e3b',3497531],
['nbs-iscc-tc:hydrangeablue.182','hydrangeablue','hydrangea blue',[67,107,149],'436b95',4418453],
['nbs-iscc-tc:imperialpurple.219','imperialpurple','imperial purple',[96,47,107],'602f6b',6303595],
['nbs-iscc-tc:independence.201','independence','independence',[37,36,64],'252440',2434112],
['nbs-iscc-tc:independence.204','independence','independence',[76,81,109],'4c516d',5001581],
['nbs-iscc-tc:indianorange.34','indianorange','indian orange',[226,88,34],'e25822',14833698],
['nbs-iscc-tc:irishgreen.142','irishgreen','irish green',[0,84,61],'00543d',21565],
['nbs-iscc-tc:ivory.92','ivory','ivory',[240,234,214],'f0ead6',15788758],
['nbs-iscc-tc:jadegreen.136','jadegreen','jade green',[103,146,103],'679267',6787687],
['nbs-iscc-tc:jasmine.86','jasmine','jasmine',[248,222,126],'f8de7e',16309886],
['nbs-iscc-tc:junglegreen.165','junglegreen','jungle green',[0,75,73],'004b49',19273],
['nbs-iscc-tc:khaki.94','khaki','khaki',[150,113,23],'967117',9859351],
['nbs-iscc-tc:lacquer.38','lacquer','lacquer',[158,71,50],'9e4732',10372914],
['nbs-iscc-tc:lavender.227','lavender','lavender',[170,152,169],'aa98a9',11180201],
['nbs-iscc-tc:leghorn.89','leghorn','leghorn',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-tc:lemonyellow.83','lemonyellow','lemon yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-tc:lightblue.173','lightblue','light blue',[54,117,136],'367588',3569032],
['nbs-iscc-tc:lightblue.186','lightblue','light blue',[83,104,120],'536878',5466232],
['nbs-iscc-tc:lightolivedrab.95','lightolivedrab','light olive drab',[108,84,30],'6c541e',7099422],
['nbs-iscc-tc:lilac.223','lilac','lilac',[134,96,142],'86608e',8806542],
['nbs-iscc-tc:limepeel.120','limepeel','limepeel ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-tc:lupine.199','lupine','lupine',[135,145,191],'8791bf',8884671],
['nbs-iscc-tc:lupine.203','lupine','lupine',[140,146,172],'8c92ac',9212588],
['nbs-iscc-tc:lustreblue.182','lustreblue','lustre blue',[67,107,149],'436b95',4418453],
['nbs-iscc-tc:magenta.256','magenta','magenta',[120,24,74],'78184a',7870538],
['nbs-iscc-tc:magenta.259','magenta','magenta',[103,49,71],'673147',6762823],
['nbs-iscc-tc:mahogany.43','mahogany','mahogany',[121,68,59],'79443b',7947323],
['nbs-iscc-tc:maize.86','maize','maize',[248,222,126],'f8de7e',16309886],
['nbs-iscc-tc:majolicablue.182','majolicablue','majolica blue',[67,107,149],'436b95',4418453],
['nbs-iscc-tc:maplesugar.77','maplesugar','maple sugar',[130,102,68],'826644',8545860],
['nbs-iscc-tc:marinecorps.183','marinecorps','marine corps',[0,48,78],'00304e',12366],
['nbs-iscc-tc:maroon.16','maroon','maroon',[114,47,55],'722f37',7483191],
['nbs-iscc-tc:maroon.259','maroon','maroon',[103,49,71],'673147',6762823],
['nbs-iscc-tc:maroon.20','maroon','maroon',[84,61,63],'543d3f',5520703],
['nbs-iscc-tc:mauve.218','mauve','mauve',[135,86,146],'875692',8869522],
['nbs-iscc-tc:melonpink.25','melonpink','melon pink',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-tc:midnight.188','midnight','midnight ',[32,40,48],'202830',2107440],
['nbs-iscc-tc:mintleaf.131','mintleaf','mintleaf ',[68,148,74],'44944a',4494410],
['nbs-iscc-tc:mistiblu.186','mistiblu','mistiblu ',[83,104,120],'536878',5466232],
['nbs-iscc-tc:myrtle.146','myrtle','myrtle ',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-tc:nationalflagblue.201','nationalflagblue','national flag blue',[37,36,64],'252440',2434112],
['nbs-iscc-tc:natural.93','natural','natural',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-tc:navy1.183','navy1','navy 1 ',[0,48,78],'00304e',12366],
['nbs-iscc-tc:navy2.183','navy2','navy 2 ',[0,48,78],'00304e',12366],
['nbs-iscc-tc:nickel.22','nickel','nickel',[143,129,127],'8f817f',9404799],
['nbs-iscc-tc:nickel.265','nickel','nickel',[132,132,130],'848482',8684674],
['nbs-iscc-tc:nile.119','nile','nile',[201,220,137],'c9dc89',13229193],
['nbs-iscc-tc:nile.120','nile','nile',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-tc:nude.79','nude','nude',[174,155,130],'ae9b82',11443074],
['nbs-iscc-tc:nuggetgold.85','nuggetgold','nugget gold',[175,141,19],'af8d13',11504915],
['nbs-iscc-tc:oakwood.58','oakwood','oakwood',[111,78,55],'6f4e37',7294519],
['nbs-iscc-tc:oldblue.185','oldblue','old blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-tc:oldchina.182','oldchina','old china',[67,107,149],'436b95',4418453],
['nbs-iscc-tc:oldgloryblue.200','oldgloryblue','old glory blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-tc:oldgloryred.11','oldgloryred','old glory red',[190,0,50],'be0032',12451890],
['nbs-iscc-tc:oldgold.88','oldgold','old gold',[171,145,68],'ab9144',11243844],
['nbs-iscc-tc:oldgold.87','oldgold','old gold',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-tc:oldrose.6','oldrose','old rose',[192,128,129],'c08081',12615809],
['nbs-iscc-tc:olive.125','olive','olive',[74,93,35],'4a5d23',4873507],
['nbs-iscc-tc:orange.50','orange','orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-tc:orange.34','orange','orange',[226,88,34],'e25822',14833698],
['nbs-iscc-tc:orchid.222','orchid','orchid',[182,149,192],'b695c0',11965888],
['nbs-iscc-tc:orientalblue.178','orientalblue','oriental blue',[0,103,165],'0067a5',26533],
['nbs-iscc-tc:paleblue.154','paleblue','pale blue',[178,190,181],'b2beb5',11714229],
['nbs-iscc-tc:paleblue.264','paleblue','pale blue',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-tc:palmetto.150','palmetto','palmetto',[94,113,106],'5e716a',6189418],
['nbs-iscc-tc:pansy.207','pansy','pansy',[96,78,151],'604e97',6311575],
['nbs-iscc-tc:pansy.208','pansy','pansy',[50,23,77],'32174d',3282765],
['nbs-iscc-tc:paprica.34','paprica','paprica',[226,88,34],'e25822',14833698],
['nbs-iscc-tc:parmaviolet.207','parmaviolet','parma violet',[96,78,151],'604e97',6311575],
['nbs-iscc-tc:parrotblue.164','parrotblue','parrot blue',[49,120,115],'317873',3242099],
['nbs-iscc-tc:parrotblue.173','parrotblue','parrot blue',[54,117,136],'367588',3569032],
['nbs-iscc-tc:pastelblue.184','pastelblue','pastel blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-tc:pastelpink.5','pastelpink','pastel pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-tc:peach.29','peach','peach',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-tc:peachblossom.3','peachblossom','peach blossom',[228,113,122],'e4717a',14971258],
['nbs-iscc-tc:peachblossom.258','peachblossom','peach blossom',[168,81,110],'a8516e',11030894],
['nbs-iscc-tc:peacock.173','peacock','peacock',[54,117,136],'367588',3569032],
['nbs-iscc-tc:pearlgrey.93','pearlgrey','pearl grey ',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-tc:pearlgrey.264','pearlgrey','pearl grey ',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-tc:peasantblue.186','peasantblue','peasant blue',[83,104,120],'536878',5466232],
['nbs-iscc-tc:pekingblue.182','pekingblue','peking blue',[67,107,149],'436b95',4418453],
['nbs-iscc-tc:pekingblue.183','pekingblue','peking blue',[0,48,78],'00304e',12366],
['nbs-iscc-tc:periwinkleblue.199','periwinkleblue','periwinkle blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-tc:pigeon.22','pigeon','pigeon',[143,129,127],'8f817f',9404799],
['nbs-iscc-tc:pimento.11','pimento','pimento',[190,0,50],'be0032',12451890],
['nbs-iscc-tc:plum.224','plum','plum',[86,60,92],'563c5c',5651548],
['nbs-iscc-tc:popcorn.87','popcorn','popcorn',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-tc:primitivegreen.141','primitivegreen','primitive green',[0,121,89],'007959',31065],
['nbs-iscc-tc:princetonorange.50','princetonorange','princeton orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-tc:prune.224','prune','prune',[86,60,92],'563c5c',5651548],
['nbs-iscc-tc:prune.225','prune','prune',[48,25,52],'301934',3152180],
['nbs-iscc-tc:purple.212','purple','purple',[47,33,64],'2f2140',3088704],
['nbs-iscc-tc:purpleorchid.237','purpleorchid','purple orchid',[158,79,136],'9e4f88',10375048],
['nbs-iscc-tc:putty.60','putty','putty',[149,128,112],'958070',9797744],
['nbs-iscc-tc:putty.63','putty','putty',[142,130,121],'8e8279',9339513],
['nbs-iscc-tc:raspberry.258','raspberry','raspberry',[168,81,110],'a8516e',11030894],
['nbs-iscc-tc:raspberry.259','raspberry','raspberry',[103,49,71],'673147',6762823],
['nbs-iscc-tc:redgrape.259','redgrape','redgrape',[103,49,71],'673147',6762823],
['nbs-iscc-tc:reseda.150','reseda','reseda',[94,113,106],'5e716a',6189418],
['nbs-iscc-tc:reseda.151','reseda','reseda',[58,75,71],'3a4b47',3820359],
['nbs-iscc-tc:riverblue.173','riverblue','river blue',[54,117,136],'367588',3569032],
['nbs-iscc-tc:riverblue.186','riverblue','river blue',[83,104,120],'536878',5466232],
['nbs-iscc-tc:robinsegg.186','robinsegg','robin\'s egg',[83,104,120],'536878',5466232],
['nbs-iscc-tc:rosebeige.57','rosebeige','rose beige',[166,123,91],'a67b5b',10910555],
['nbs-iscc-tc:rosepink.2','rosepink','rose pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-tc:royalblue.183','royalblue','royal blue',[0,48,78],'00304e',12366],
['nbs-iscc-tc:royalblue.179','royalblue','royal blue',[0,65,106],'00416a',16746],
['nbs-iscc-tc:ruby.256','ruby','ruby',[120,24,74],'78184a',7870538],
['nbs-iscc-tc:salmon.37','salmon','salmon',[203,109,81],'cb6d51',13331793],
['nbs-iscc-tc:salmonpink.26','salmonpink','salmon pink ',[248,131,121],'f88379',16286585],
['nbs-iscc-tc:sand.93','sand','sand',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-tc:sandalwood.57','sandalwood','sandalwood',[166,123,91],'a67b5b',10910555],
['nbs-iscc-tc:sapphireblue.197','sapphireblue','sapphire blue',[39,36,88],'272458',2565208],
['nbs-iscc-tc:saxeblue.186','saxeblue','saxe blue',[83,104,120],'536878',5466232],
['nbs-iscc-tc:scarabgreen.140','scarabgreen','scarab green',[62,180,137],'3eb489',4109449],
['nbs-iscc-tc:scarlet.11','scarlet','scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-tc:scarlet.12','scarlet','scarlet',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-tc:seal.62','seal','seal',[62,50,44],'3e322c',4076076],
['nbs-iscc-tc:seapink.3','seapink','sea pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-tc:shellpink.26','shellpink','shell pink',[248,131,121],'f88379',16286585],
['nbs-iscc-tc:shrimppink.3','shrimppink','shrimp pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-tc:shrimppink.27','shrimppink','shrimp pink',[230,103,97],'e66761',15099745],
['nbs-iscc-tc:silver.265','silver','silver',[132,132,130],'848482',8684674],
['nbs-iscc-tc:silvergray.265','silvergray','silver gray',[132,132,130],'848482',8684674],
['nbs-iscc-tc:sistine.185','sistine','sistine',[145,163,176],'91a3b0',9544624],
['nbs-iscc-tc:skyblue.185','skyblue','sky blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-tc:skyblue.183','skyblue','sky blue',[0,48,78],'00304e',12366],
['nbs-iscc-tc:smoke.266','smoke','smoke',[85,85,85],'555555',5592405],
['nbs-iscc-tc:spanishyellow.68','spanishyellow','spanish yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-tc:spicebrown.58','spicebrown','spicebrown ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-tc:spraygreen.149','spraygreen','spraygreen ',[141,163,153],'8da399',9282457],
['nbs-iscc-tc:springgreen.117','springgreen','spring green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-tc:starlightblue.203','starlightblue','starlight blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-tc:steel.233','steel','steel',[139,133,137],'8b8589',9143689],
['nbs-iscc-tc:stoneblue.186','stoneblue','stone blue',[83,104,120],'536878',5466232],
['nbs-iscc-tc:strawberry.258','strawberry','strawberry',[168,81,110],'a8516e',11030894],
['nbs-iscc-tc:sunset.73','sunset','sunset',[250,214,165],'fad6a5',16438949],
['nbs-iscc-tc:tan.76','tan','tan',[193,154,107],'c19a6b',12687979],
['nbs-iscc-tc:tangerine.35','tangerine','tangerine',[217,96,59],'d9603b',14245947],
['nbs-iscc-tc:tarragon.136','tarragon','tarragon',[103,146,103],'679267',6787687],
['nbs-iscc-tc:taupe.64','taupe','taupe',[91,80,79],'5b504f',5984335],
['nbs-iscc-tc:teal.174','teal','teal',[0,73,88],'004958',18776],
['nbs-iscc-tc:tearose.26','tearose','tea rose',[248,131,121],'f88379',16286585],
['nbs-iscc-tc:terracotta.38','terracotta','terra cotta',[158,71,50],'9e4732',10372914],
['nbs-iscc-tc:terracotta.40','terracotta','terra cotta',[136,45,23],'882d17',8924439],
['nbs-iscc-tc:terracotta.43','terracotta','terra cotta',[121,68,59],'79443b',7947323],
['nbs-iscc-tc:tigerlily.27','tigerlily','tigerlily',[230,103,97],'e66761',15099745],
['nbs-iscc-tc:tileblue.186','tileblue','tile blue',[83,104,120],'536878',5466232],
['nbs-iscc-tc:toastbrown.58','toastbrown','toast brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-tc:tobacco.58','tobacco','tobacco',[111,78,55],'6f4e37',7294519],
['nbs-iscc-tc:topaz.72','topaz','topaz',[190,138,61],'be8a3d',12487229],
['nbs-iscc-tc:tourmaline.148','tourmaline','tourmaline',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-tc:turquoise.163','turquoise','turquoise',[102,173,164],'66ada4',6729124],
['nbs-iscc-tc:ultramarineblue.176','ultramarineblue','ultramarine blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-tc:ultramarineblue.194','ultramarineblue','ultramarine blue',[48,38,122],'30267a',3155578],
['nbs-iscc-tc:vassarrose.250','vassarrose','vassar rose',[213,151,174],'d597ae',13997998],
['nbs-iscc-tc:violet.207','violet','violet',[96,78,151],'604e97',6311575],
['nbs-iscc-tc:violinepink.258','violinepink','violine pink',[168,81,110],'a8516e',11030894],
['nbs-iscc-tc:westpoint.186','westpoint','west point',[83,104,120],'536878',5466232],
['nbs-iscc-tc:white.92','white','white',[240,234,214],'f0ead6',15788758],
['nbs-iscc-tc:white.263','white','white',[242,243,244],'f2f3f4',15922164],
['nbs-iscc-tc:white.92','white','white',[240,234,214],'f0ead6',15788758],
['nbs-iscc-tc:white.93','white','white',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-tc:yaleblue.197','yaleblue','yale blue',[39,36,88],'272458',2565208],
['nbs-iscc-tc:yellow.68','yellow','yellow',[234,162,33],'eaa221',15376929]
    ];
}

sub _description {
    return {
          'subtitle' => 'Standard Color Card of America',
          'title' => 'NBS/ISCC TC',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (TC) Standard Color Card of America

    Textile Color Card Association [TCCA],
    Standard Color Card of America and U.S. Army Color Card 

The 237 names of this idiosyncratic dictionary map to 267 colors, but only 129 out of 267 possible centroids are called out.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#TC]
'
        }

}

1;
