#!/usr/local/bin/perl -w
#
# $Id: Makefile.PL 11264 2008-05-14 12:08:27Z mjevans $
#
# You may distribute under the terms of either the GNU General Public
# License or the Artistic License, as specified in the Perl README file.
#
# Test dependencies on CPAN:
# http://cpandeps.cantrell.org.uk/?module=DBD::ODBC;perl=latest
#
BEGIN { require 5.004 }	# 5.004 is required for Win32
use Config;
use ExtUtils::MakeMaker 5.16, qw(&WriteMakefile $Verbose prompt);
use File::Basename;
use Getopt::Long;
use File::Spec;

use strict;

$| = 1;
$^W = 1;
my %opts =
(
    NAME	=> 'DBD::ODBC',
    VERSION_FROM => 'ODBC.pm',
    NO_META => 1,
    # See note below on CONFIGURE. This used to work when could rely on
    # CONFIGURE being after PREREQ_PM but that is not the case now so the
    # line below does nothing since 6.33 of MakeMaker.
    PREREQ_PM => {"Test::Simple" => 0.40,# actually Test::More pkg in T::S dist,
                  "DBI" => 1.21 },
    clean	=> { FILES=> 'ODBC.xsi dbdodbc.h' },
    dist	=> {
	DIST_DEFAULT=> 'clean distcheck tardist',
	PREOP    => '$(MAKE) -f Makefile.old distdir',
	COMPRESS => 'gzip -v9', SUFFIX => 'gz'
    },
    OBJECT	=> '$(O_FILES)',
    DEFINE => ''
 );
my $eumm = $ExtUtils::MakeMaker::VERSION;
$eumm =~ tr/_//d;
if ($eumm >= 5.43) {
    $opts{AUTHOR} = 'Tim Bunce, Jeff Urlwin, Martin J. Evans mailto:dbi-users@perl.org';
    $opts{ABSTRACT} = 'ODBC driver for the DBI module.';
    $opts{CAPI} = 'TRUE' if $Config{archname} =~ /-object\b/i;
    $opts{PREREQ_PRINT} = 1;
    #$opts{PREREQ_FATAL} = 1;
    # See
    # http://www.mail-archive.com/cpan-testers-discuss%40perl.org/msg00076.html
    # In ExtUtils::MakeMaker 6.32 and earlier CONFIGURE was run after PREREQ_PM
    # so we could safely require DBI::DBD here and PREREQ_PM would fail first
    # if DBI was not installed. Since 6.33 CONFIGURE is run before PREREQ_PM
    # so now the require below fails and if we do not exit 0 without generating
    # a Makefile cpan-testers will fail us if DBI is not found.
    $opts{CONFIGURE} = sub {
        eval {require DBI::DBD;};
        if ($@) {
            warn $@;
            exit 0;
        } else {
        my $dbi_arch_dir = DBI::DBD::dbd_dbi_arch_dir();
        if (exists($opts{INC})) {
            return {INC => "$opts{INC} -I$dbi_arch_dir"};
        } else {
            return {INC => "-I$dbi_arch_dir"};
        }
    }
    };
}

my $opt_g = 0;
my $opt_o = '';
my $opt_u = undef;

Getopt::Long::GetOptions("g!" => \$opt_g,
                         "o=s" => \$opt_o,
                         "u!" => \$opt_u) or die "Invalid arguments\n";
if (($ENV{LANG} || '') =~ m/utf-?8/i) {
    print <<"EOF";

Your LANG environment variable is set to "$ENV{LANG}"\a
This is known to cause problems in some perl installations - even stopping
this Makefile.PL to run without errors. If you have problems please try
re-running with LANG unset or with the utf part of LANG removed.

EOF
    sleep 4;
}

if ($Config{useithreads}) {
    print <<"EOF";
You are using a Perl configured with threading enabled\a.
Please read the warnings in DBI about this.
You should also be aware that on non-Windows platforms ODBC drivers come
in two forms, thread-safe and non-thread-safe drivers and you may need
to make sure you are using the right one.

EOF
    prompt("Press return to continue...");
}
print "Overriding ODBC Directory with command line option: $opt_o\n"
    if $opt_o ;
if ($opt_g) {
   print "Setting debug options!\n";
   if ($^O eq 'MSWin32') {
      $opts{OPTIMIZE} = '/Zi';
   } else {
      $opts{OPTIMIZE} = '-g -O0';
   }
}

if (defined($opt_u) && $opt_u) {
    $opts{DEFINE} .= ' -DWITH_UNICODE';
    warn "\n\n\nWarning:Building with UNICODE enabled is experimental at present\a\n\n\n";
    prompt("Press return to continue...");
    require 5.008001;
}

print <<"EOT";

>>>\tRemember to actually *READ* the README file!
   \tAnd re-read it if you have any problems.\n

EOT

open(SQLH, ">dbdodbc.h") || die "Can't open dbdodbc.h: $!\n";
print SQLH "/* Do not edit this file. It is automatically written by Makefile.PL.\n";
print SQLH "   Any changes made here will be lost. \n*/\n\n";
print SQLH "#undef WORD /* from perly.y */\n";

my (@known_drivers) = sort { $a cmp $b } (
	'Microsoft ODBC',
	'unixodbc',
	'iodbc',
	'esodbc',
	'empress',
	'intersolve',
	'sapdb',
	'adabas',
	'udbc',
	'easysoft',
	'solid',
	'informix',
	);

if ($^O eq 'MSWin32') {
    my $extrainc = "";
    $extrainc = ";$Config{incpath}\\mfc" if $Config{cc} eq 'bcc32';
    $opts{SKIP} = ['processPL'];
    if (!defined($opt_u) || $opt_u) {
	$opts{DEFINE}  .= " -DWITH_UNICODE";
	require 5.008001;
    }
    $opts{INC}  = "$extrainc";
    $opts{LIBS} = ["ODBC32.LIB"];
    $opts{macro}->{EXTRALIB} = 'ODBC32.LIB';
    print SQLH "#include <windows.h>\n";
    print SQLH "#include <sql.h>\n#include <sqltypes.h>\n#include <sqlext.h>\n";
}
else {

    # for Adabas
    $ENV{ODBCHOME} = $ENV{DBROOT} if $ENV{DBROOT} && -f "$ENV{DBROOT}/lib/odbclib.a";

    print "Overriding ODBC Directory with command line option: $opt_o\n" if $opt_o ;
    my $odbchome= $opt_o || $ENV{ODBCHOME};

    $odbchome = VMS::Filespec::unixify($odbchome) if $^O eq 'VMS';


    # if we haven't got odbchome set try and find a driver, driver manager.
    if (!$odbchome) {
        if ($ENV{WINDIR} && $^O eq 'cygwin') {
            # per patches from Teun Burgers
            my $tmp_odbchome = $ENV{WINDIR};
            $tmp_odbchome =~ s/^([A-Za-z]):*$/\/\/$1/;
            $tmp_odbchome =~ s/\\/\//g;
            $odbchome = $tmp_odbchome if (-e "$tmp_odbchome/odbc.ini")
        } elsif (-f '/opt/sapdb/interfaces/odbc/lib/libsqlod.a') {
            $odbchome = '/opt/sapdb/interfaces/odbc/';
        } elsif ($^O !~ /MSWin/) {
            # unixODBC - would have liked to use odbc_config but it did not
            # exist until 2.2.11 and it was broken wrt --cflags in 2.2.11/2.2.12
            # i.e. --cflags did not include -I/xxx/yyy
            if (`odbc_config --version 2>&1` =~ /^(\d\.)+/) {
                $odbchome = `odbc_config --prefix 2>&1`;
                chomp $odbchome;
                if ((-e $odbchome) && (-e "$odbchome/include") &&
                        (-f "$odbchome/include/sql.h") &&
                            (-f "$odbchome/include/sqlext.h") &&
                                (-f "$odbchome/include/sqltypes.h")) {
                    print "Set ODBC HOME to $odbchome from odbc_config\n";
                } else {
                    $odbchome = '';
                }
            }
            if (!$odbchome) {
                if (-f '/usr/include/sql.h' &&
                        -f '/usr/include/sqlext.h' &&
                            -f '/usr/include/sqltypes.h') {
                    $odbchome = '/usr';
                } elsif (-f '/usr/local/include/sql.h' &&
                             -f '/usr/local/include/sqlext.h' &&
                                 -f '/usr/local/include/sqltypes.h') {
                    $odbchome = '/usr/local';
                } elsif (-d '/usr/local/easysoft/unixODBC') {
                    $odbchome = '/usr/local/easysoft/unixODBC';
                }
            }
        }
        print "Found what looks like a driver manager in $odbchome\n"
            if $odbchome;
    }

    unless ($odbchome) {
	print <<'EOT';

The DBD::ODBC module needs to link with an ODBC 'Driver Manager'.
(The Driver Manager, in turn, needs one or more database specific ODBC
drivers. The DBD::ODBC module does _not_ include any ODBC drivers!)

You need to indicate where your ODBC Driver Manager is installed.
You can do this ether by setting the ODBCHOME environment variable
or by running 'perl Makefile.PL -o odbcdir'.

If you do not have an ODBC Driver Manager you should try to get hold of
the unixODBC packages for your system or build it from source (see
http://www.unixodbc.org).
EOT
        # stop cpan testers from reporting a failure when a driver manager
        # library is not installed. Do not know if Devel::CheckLib (when
        # it is released) is going to help here.
        # see http://www.mail-archive.com/cpan-testers-discuss%40perl.org/msg00043.html
        exit 0;
    }
    die "ODBCHOME environment variable ($odbchome) does not refer to a directory.\n"
	unless -d $odbchome;
    warn "Using ODBC in $odbchome\n";
    $opts{INC}  = "-I.";

    # cygwin patch
    $opts{INC}  .= " -I/usr/include/w32api" if $^O eq 'cygwin';

    # Try to work out which driver manager is being used.
    # Feel free to come up with neat (or un-neat) hacks to get your's to build!

    my $lib_d1 = "$odbchome/lib";
    my $lib_d2 = "$odbchome/dlls";
    my $libs   = "odbc";
    $opts{LIBS} = " -L$lib_d1 -R$lib_d1 -L$lib_d2 -R$lib_d2 -l$libs";


    my $myodbc ='';
    my $myodbc_version = -1;
    $myodbc = 'Microsoft ODBC'
                 if (    -e "$odbchome/system/odbc32.dll" or
			 -e "$odbchome/system32/odbc32.dll" or
                         -e "$odbchome/odbc32.dll");

    my $dlext = $Config{dlext};
    my $arext = $Config{lib_ext};

    # fix to avoid foo..ext on many systems.
    $arext =~ s/^\.//;

    $myodbc = 'unixodbc'
		if !$myodbc && <$odbchome/lib/libodbc.*>;

    $myodbc = 'iodbc'
		if !$myodbc && (<$odbchome/*iodbc*> || <$odbchome/lib/*iodbc*>);

    $myodbc = 'empress'
		if !$myodbc && <$odbchome/lib/libempodbc.*>;

    $myodbc = 'intersolve'
		if !$myodbc && -f "$odbchome/include/qeodbc.h";

    $myodbc = 'sapdb'
		if !$myodbc && -f "$odbchome/lib/libsqlod.$arext";

    $myodbc = 'adabas'
		if !$myodbc && $ENV{DBROOT} && $odbchome eq $ENV{DBROOT} && -f "$odbchome/lib/odbclib.$arext";

    $myodbc = 'udbc'
		if !$myodbc && -f "$odbchome/lib/libudbc.$arext";

    $myodbc = 'solid'
		if !$myodbc && -f "$odbchome/lib/libsolcli.$dlext";

    # JL 2002-12-16: This test is accurate on Unix (Solaris 7) with IBM
    # Informix ClientSDK 2.80.UC1, which includes IBM Informix CLI
    # v3.81.000, an ODBC 3.x driver.
	# NB: The correct value for $ODBCHOME is $INFORMIXDIR.
    $myodbc = 'informix'
		if !$myodbc && -f "$odbchome/lib/cli/libifcli.$dlext";

    if (!$myodbc) {
	local($") = ", ";
	my($list) = "@known_drivers";
	$list =~ s%^(.{30,70})\s%$1\n\t%gmo;
	die qq%
I cannot find an ODBC driver manager that I recognize.
...And I know about these drivers:
	$list
%;
    }

    warn "\nThis looks like a $myodbc type of driver manager.\n";

    if ($myodbc eq 'Microsoft ODBC') {
	print "\nBuilding for Microsoft under Cygwin\n";
	$opts{LIBS} = "-L/usr/lib/w32api -lodbc32";
        print SQLH "#include <windows.h>\n";
        print SQLH "#include <sql.h>\n";
        print SQLH "#include <sqltypes.h>\n";
        print SQLH "#include <sqlext.h>\n";
        print SQLH "#undef WIN32\n";
        $opts{dynamic_lib} = {OTHERLDFLAGS => "-lodbc32"};
    }
    elsif ($myodbc eq 'iodbc') {
	my $ilibdir = "$odbchome/lib";
	my @ilibs = <$ilibdir/*iodbc*.*>;
	@ilibs = grep { /\.($Config{so}|$Config{dlext}|a)$/ } @ilibs;
	die "That's odd, I can't see any iodbc libs in $ilibdir" unless @ilibs;
	# This is a hack using Intersolve's free Solaris ODBC manager
	# Since it doesn't come with the sql header files (!) we get them from iODBC!
	# Note: we use DEFINE not INC for iODBC so we don't get its config.h
	print "\n";
	print "We expect to find the isql.h, isqlext.h and iodbc.h files (which were\n";
	print "supplied with iODBC) in \$ODBCHOME/include directory alongside\n";
	print "the @ilibs library.\n\n";
	my $ilibpath = $ilibs[0]; # XXX if both .so and .a, pick based on LINKTYPE?
	my $ilibname = basename($ilibpath);
	$opts{DEFINE} .= " -I$odbchome/include";
	if ($ilibname =~ /^iodbc/) { # no "lib" prefix
	    $opts{LIBS} = "";
	    $opts{dynamic_lib} = { OTHERLDFLAGS => "$ilibpath" };
	}
	else {
	    # remove lib prefix and .so suffix so "-l" style link can be used
	    $ilibname =~ s/^lib(iodbc.*?)\.\w+$/$1/;
	    $opts{LIBS} = "-L$ilibdir -l$ilibname";

	    # changed /\Q$ilibpath/ to /\Q$ilibdir/ per recommendation
	    # by Ron Savage
	    warn "Warning: LD_LIBRARY_PATH doesn't include $odbchome/lib\n"
		unless $ENV{LD_LIBRARY_PATH} =~ /\Q$ilibdir/;
	}
	#print SQLH qq{#define FAR \n#define EXPORT \n#define CALLBACK \n};
	#print SQLH qq{#include <iodbc.h>\n};
	print SQLH qq{#include <isqlext.h>\n};
	print SQLH qq{#include <sql.h>\n};
	print SQLH qq{#include <sqltypes.h>\n};
	#print SQLH qq{#include "fixup_t.h"\n};
    }
    elsif ($myodbc eq 'unixodbc') {
        # if we find odbcinst, output useful info about this version of unixODBC
        # and store unixODBC version
        if (-x "$odbchome/bin/odbcinst") {
            my $j = `$odbchome/bin/odbcinst -j 2>&1`;
            print "odbcinst -j reports:\n\n$j\n" if $j;
            if ($j =~ /^unixODBC ([\d\.]+).*/ ) {
                $myodbc_version = $1;
            }
        }
        # if we find odbc_config add --cflags output to CC line
        if (-x "$odbchome/bin/odbc_config") {
            my @args = qw(--prefix --include-prefix --lib-prefix --version --odbcversion);
            for my $oca (@args) {
                my $c = `$odbchome/bin/odbc_config $oca 2>&1`;
                chomp $c;
                if ($c) {
                    print "odbc_config $oca = $c\n";
                }
            }
            print "\n";
            my $cf = `$odbchome/bin/odbc_config --cflags 2>&1`;
            if ($cf =~ /\-D/) {
                chomp $cf;
                $cf =~ s/\n//g;
                print qq/Adding odbc_config --cflags "$cf" to CC line\n/;
                $opts{DEFINE} .= " $cf";
            }
        }
	my @ilibs = <$odbchome/lib/libodbc.*>;
	@ilibs = grep { /\.($Config{so}|$Config{dlext}|a)$/ } @ilibs;
	die "That's odd, I can't see any unixodbc libs in $odbchome" unless @ilibs;
	print << "EOT";

We expect to find the sql.h, sqlext.h, sqltypes.h and sqlucode.h (which were
supplied with unixODBC) in \$ODBCHOME/include directory alongside the
@ilibs library in \$ODBCHOME/lib

EOT
	my $ilibpath = $ilibs[0]; # XXX if both .so and .a, pick based on LINKTYPE?
	my $ilibname = basename($ilibpath);
	$opts{DEFINE} .= " -I$odbchome/include";
	if ($ilibname =~ /^odbc/) { # no "lib" prefix
	    $opts{LIBS} = "";
	    $opts{dynamic_lib} = { OTHERLDFLAGS => "$ilibpath" };
	}
	else {
	    # remove lib prefix and .so suffix so "-l" style link can be used
	    $ilibname =~ s/^lib(odbc.*?)\.\w+$/$1/;
	    $opts{LIBS} = "-L$odbchome/lib -l$ilibname";
	    warn "Warning: LD_LIBRARY_PATH doesn't include $odbchome\n"
		unless (exists($ENV{LD_LIBRARY_PATH}) &&
                        ($ENV{LD_LIBRARY_PATH} =~ /\Q$odbchome\/lib/));
	}
	print SQLH qq{#include <sql.h>\n};
	print SQLH qq{#include <sqlucode.h>\n};
	print SQLH qq{#include <sqltypes.h>\n};
	print SQLH qq{#include <sqlext.h>\n};
    }
    elsif ($myodbc eq 'esodbc') {
	my @ilibs = <$odbchome/*esoobclient.*>;
	print "Located library @ilibs\n";
	# omit . prefix in next grep as some $Config types have . and some don't
	@ilibs = grep { /($Config{so}|$Config{dlext}|$Config{lib_ext})$/ } @ilibs;
	die "That's odd, I can't see any esoobclient libs in $odbchome" unless @ilibs;
	print "We expect to find the sql.h and sqlext.h files (which were\n";
	print "supplied with esoobclient) in \$ODBCHOME/include directory alongside\n";
	print "the @ilibs library.\n\n";
	my $ilibpath = $ilibs[0]; # XXX if both .so and .a, pick based on LINKTYPE?
	my $ilibname = basename($ilibpath);
	print "Using library $ilibname\n";
	$opts{INC} .= " -I$odbchome/include";
	#
	# ptrmismatch avoids the char/unsigned problems in DBD
	# outtyplen is one incorrect fprintf in dbiimp.c(200)
	# promotmatchw - old prototypes problem
	#
	$opts{CCFLAGS} .= "/warnings=informational=(outtypelen,ptrmismatch,promotmatchw)"
	    if ($Config{cc} =~ /DECC/ && $^O eq 'VMS' );
	if ($ilibname =~ /^esoobclient/) { # no "lib" prefix
	    $opts{LIBS} = "";
	    $opts{dynamic_lib} = { OTHERLDFLAGS => "$ilibpath" };
	}
	else {
	    my $extralibs="";
	    if (($^O eq 'VMS') && ($ilibname =~ /$Config{lib_ext}$/)) {
		    $extralibs = " -lcs_share -lsupport -leasyrpc";
		    print "Adding libraries $extralibs\n";
	    }
	    # remove lib prefix and so suffix so "-l" style link can be used
	    $ilibname =~ s/^lib(esoobclient.*?)\.\w+$/$1/;
	    $opts{LIBS} = "-L$odbchome -l$ilibname $extralibs";
	    if ($^O ne "VMS") {
	       warn "Warning: LD_LIBRARY_PATH doesn't include $odbchome\n"
		     unless $ENV{LD_LIBRARY_PATH} =~ /\Q$odbchome/;
	    }
	}
	print SQLH qq{#define FAR \n#define EXPORT \n#define CALLBACK \n};
	print SQLH qq{#include <sqlext.h>\n};
	print SQLH qq{#include <sqlucode.h>\n};
	print SQLH qq{#include <sql.h>\n};
	print SQLH qq{#include <sqltypes.h>\n};
    }
    elsif ($myodbc eq 'intersolve') {
	#$opts{DEFINE}  .= "";
  	print SQLH qq{#include <qeodbc.h>\n};
	if (-f "$odbchome/include/sql.h") {
	    print "You seem to have the official header files.\n";
	    $opts{INC} .= " -I$odbchome/include";
	    print SQLH qq{#include <sql.h>\n#include <sqltypes.h>\n#include <sqlext.h>\n};
	}
	else {
	    # This is common on Solaris
	    print "You don't seem to have the official header files,\n";
	    print "so I'll use the iODBC ones instead.\n";
	    $opts{INC} .= " -I$odbchome/include -Iiodbcsrc";
	    print SQLH qq{#include <isql.h> \n#include <isqlext.h>\n};
	}
    }
    elsif ($myodbc eq 'empress') {
	$opts{INC} .= " -I$odbchome/include";
	#$opts{DEFINE}  .= "";
	print SQLH qq{#include <odbcsys.h>\n};
	print SQLH qq{#include <sql.h>\n#include <sqlext.h>\n};
	$opts{LIBS} = "-L$odbchome/lib -R$odbchome/lib -lempodbc";
    }
    elsif ($myodbc eq 'sapdb') {
	print SQLH "#include <WINDOWS.H>\n";
	print SQLH "#include <sql.h>\n";
	print SQLH "#include <sqlext.h>\n";
	print SQLH "#define HENV SQLHENV\n";
	print SQLH "#define HDBC SQLHDBC\n";
	print SQLH "#define HSTMT SQLHSTMT\n";
	print SQLH "#define DBD_ODBC_NO_SQLDRIVERCONNECT\n";
	print SQLH qq{#define DBD_ODBC_NO_DATASOURCES\n}; # unless ($^O eq 'MSWin32');

	$opts{INC} .= " -I$odbchome/incl";
	$opts{LDFROM} = "\$(OBJECT) $odbchome/lib/libsqlod.a";
    }
    elsif ($myodbc eq 'adabas') {
	print SQLH "#define FAR \n#define EXPORT \n#define CALLBACK \n";
	print SQLH "#include <WINDOWS.H>\n";
	print SQLH "#include <sql.h>\n";
	print SQLH "#include <sqlext.h>\n";
	$opts{INC} .= " -I$odbchome/incl";
	$opts{LIBS} = "-L$odbchome/lib -lsqlrte -lsqlptc";
	$opts{LDFROM} = "\$(OBJECT) $odbchome/lib/odbclib.a";
    }
    elsif ($myodbc eq 'udbc') {
        print SQLH qq{#include <libudbc.h>\n};
        $opts{INC} .= " -I$odbchome/include";
        #$opts{DEFINE}  .= "";
        $opts{LIBS} = "-L$odbchome/lib -R$odbchome/lib -ludbc";
    }
    elsif ($myodbc eq 'easysoft') {
        $opts{INC} .= " -I$odbchome/include";
        $opts{LIBS} = "-L$odbchome/lib -lesoobclient";
        print SQLH qq{#include <sql.h>\n#include <sqlext.h>\n};
    }
    elsif ($myodbc eq 'solid') {
        $opts{INC} .= " -I$odbchome/include";
        $opts{LIBS} = "-L$odbchome/lib -lsolcli";
	# Solid does not support DataSources
	print SQLH qq{#define DBD_ODBC_NO_DATASOURCES\n};
	# Solid does not support DataSources
	print SQLH qq{#define DBD_ODBC_NO_SQLDRIVERCONNECT\n};
        print SQLH qq{#include <cli0cli.h>\n};
    }
    elsif ($myodbc eq 'informix') {
		# JL 2002-12-16: See comments above for environment details.
        $opts{INC}  = "-I$odbchome/incl/cli $opts{INC}";
        $opts{LIBS} = "-L$odbchome/lib/cli -lifcli -lifdmr";
        $opts{DEFINE} .= " -DNO_WIN32";		# Applies to Unix only, of course
        print SQLH qq{#include <stddef.h>\n};
        print SQLH qq{#include <infxcli.h>\n};
    }
    else {
	print <<'EOT';
*** WARNING ***
Unknown driver manager. Using default build process.
This will almost certainly fail at some point.
In which case you will need to edit/hack the Makefile.PL
to suit your needs. (Specifically to locate your odbc
library and header files.)
EOT
	#$opts{DEFINE}  = "";
	print SQLH qq{#include <sql.h> \n#include <sqlext.h>\n};
    }
}
print SQLH qq{\n};
print SQLH qq{#include "fixup_c.h"\n};
print SQLH qq{\n};
close(SQLH);

print "\n";

my $rv = WriteMakefile(%opts);

local($^W)=0;
print qq{
The DBD::ODBC tests will use these values for the database connection:
    DBI_DSN=$ENV{DBI_DSN}		e.g. dbi:ODBC:demo
    DBI_USER=$ENV{DBI_USER}
    DBI_PASS=$ENV{DBI_PASS}
};
print "Warning: not all required environment variables are set.\n"
	unless ($ENV{DBI_DSN} && $ENV{DBI_USER} && $ENV{DBI_PASS});
print "Warning: DBI_DSN ($ENV{DBI_DSN}) doesn't start with 'dbi:ODBC:'\n"
	if ($ENV{DBI_DSN} && $ENV{DBI_DSN} !~ m/^dbi:ODBC:/);
print "\n";

# Following generates
# Useless use of private variable in void context at Makefile.PL
# but is required
$rv;

# ====================================================================

package MY;

use strict;
use Config;


sub postamble {
    return DBI::DBD::dbd_postamble(@_);
}

sub const_cccmd {
    my $self = shift;
    local($_) = $self->SUPER::const_cccmd(@_);

    # inject the defined local ODBC before default include to ensure
    # the ODBC driver we want is first
    if ($^O ne 'MSWin32') {
        s/-c/-c \$\(DEFINE\)/;
    }
    $_;
}
sub MY::post_constants {
    my ($self) = shift;

    '
# make Changes file available as installed pod docs "perldoc DBD::ODBC::Changes"
inst_libdbdodbc = ' . File::Spec->catdir($self->{INST_LIB}, 'DBD/ODBC') . '
changes_pm = ' . File::Spec->catfile($self->{INST_LIB}, 'DBD/ODBC', 'Changes.pm') . '

# make FAQ file available as installed pod docs "perldoc DBD::ODBC::FAQ"
inst_libdbdodbc = ' . File::Spec->catdir($self->{INST_LIB}, 'DBD/ODBC') . '
faq_pm = ' . File::Spec->catfile($self->{INST_LIB}, 'DBD/ODBC', 'FAQ.pm') . '

config :: $(changes_pm) $(faq_pm)
	@$(NOOP)

$(changes_pm): Changes
	$(NOECHO) $(MKPATH) $(inst_libdbdodbc)
	$(NOECHO) $(RM_F) $(changes_pm)
	$(CP) Changes $(changes_pm)

$(faq_pm): FAQ
	$(NOECHO) $(MKPATH) $(inst_libdbdodbc)
	$(NOECHO) $(RM_F) $(faq_pm)
	$(CP) FAQ $(faq_pm)
';

}

# end.


