use 5.008;
use strict;
use warnings;

package Bash::Completion::Plugins::VimTag;
BEGIN {
  $Bash::Completion::Plugins::VimTag::VERSION = '1.110690';
}
# ABSTRACT: Bash completion plugin for tags generated by Vim::Tag

use parent 'Bash::Completion::Plugin';

use Bash::Completion::Utils qw(command_in_path);

sub should_activate {
    [grep { command_in_path($_) } qw(ptags)];
}

sub generate_bash_setup {
    q{complete -C 'bash-complete complete VimTag' -o nospace -o default vit};
}

sub complete {
    my ($self, $r) = @_;
    my $word = $r->word;
    open my $fh, '<', $ENV{PTAGSFILE} or return; # fail silently

    # The tags file is sorted alphabetically, so if we've found something,
    # then from that point on, if a line doesn't match, we don't need to
    # search any further - we can't match anything that comes later in the
    # file either.
    #
    # Compared to slurping the whole file, then checking every line, this is a
    # lot faster, so the bash completion feels a lot more responsive.

    my $has_hits = 0;
    my @candidates;
    while (<$fh>) {
        if (index($_, $word) == 0) {
            push @candidates, (split)[0];
            $has_hits++;
        } else {
            last if $has_hits;
        }
    }
    close $fh or return;

    # Special case if the word contains one or more colons: Only return the
    # part of each completion that comes after the last colon. This has to do
    # with bash using the colon as a delimiter, I guess.

    if ($word =~ /^(.+::?)/) {
        my $prefix_length = length $1;
        substr($_, 0, $prefix_length) = '' for @candidates;
    }
    $r->candidates(@candidates);
}

1;


__END__
=pod

=head1 NAME

Bash::Completion::Plugins::VimTag - Bash completion plugin for tags generated by Vim::Tag

=head1 VERSION

version 1.110690

=head1 DESCRIPTION

This is a L<Bash::Completion> plugin that can complete on tags generated by
the C<ptags> program that is included in the L<Vim-Tag> distribution. It
completes on the C<vit> command, which you can use as a bash alias for C<vi
-t>.

=head1 METHODS

=head2 should_activate

Says that this plugin should activate if C<ptags> is in the path.

=head2 generate_bash_setup

Defines that this plugin should be used for the C<vit> command or bash alias.

Why does C<should_activate()> use a different command? C<ptags> is the program
that generates the tags, but C<vit> (alias for C<vi -t>) is the command for
which we want to complete.

=head2 complete

Takes a request and returns possible candidates from the file specified by
C<$ENV{PTAGSFILE}>.

=head1 PLANS

=over 4

=item * Completing only on C<vit> and expecting it to be a bash alias for C<vi
-t> is too restrictive.

=back

=head1 INSTALLATION

See perlmodinstall for information and options on installing Perl modules.

=head1 BUGS AND LIMITATIONS

No bugs have been reported.

Please report any bugs or feature requests through the web interface at
L<http://rt.cpan.org/Public/Dist/Display.html?Name=Bash-Completion-Plugins-VimTag>.

=head1 AVAILABILITY

The latest version of this module is available from the Comprehensive Perl
Archive Network (CPAN). Visit L<http://www.perl.com/CPAN/> to find a CPAN
site near you, or see L<http://search.cpan.org/dist/Bash-Completion-Plugins-VimTag/>.

The development version lives at L<http://github.com/hanekomu/Bash-Completion-Plugins-VimTag>
and may be cloned from L<git://github.com/hanekomu/Bash-Completion-Plugins-VimTag.git>.
Instead of sending patches, please fork this project using the standard
git and github infrastructure.

=head1 AUTHOR

Marcel Gruenauer <marcel@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2011 by Marcel Gruenauer.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

