#!/usr/bin/perl -w
#$Id: capture.t,v 1.3 2004/11/22 19:51:09 simonflack Exp $
use strict;
use Test::More tests => 15;
use IO::CaptureOutput 'capture';

my ($out, $err);
sub _reset { $_ = '' for ($out, $err); 1};

# Basic test
_reset && capture sub {print __PACKAGE__; print STDERR __FILE__}, \$out, \$err;
is($out, __PACKAGE__, 'captured stdout from perl function');
is($err, __FILE__, 'captured stderr from perl function');

# merge STDOUT and STDERR
_reset && capture sub {print __PACKAGE__; print STDERR __FILE__}, \$out, \$out;
like($out, q{/^} . quotemeta(__PACKAGE__) . q{/}, 
    'captured stdout into one scalar');
like($out, q{/} . quotemeta(__FILE__) . q{$/}, 
    'captured stderr into same scalar');

# Check we still get return values
_reset;
my @arg = capture sub {print 'Testing'; return (1,2,3)}, \$out, \$err;
ok($out eq 'Testing' && eq_array(\@arg, [1,2,3]),
   'capture() proxies the return values');

# Check that the captured sub is called in the right context
my $context = capture sub {wantarray};
ok(defined $context && ! $context,
   'capture() calls subroutine in scalar context when appropriate');

($context) = capture sub {wantarray};
ok($context, 'capture() calls subroutine in list context when appropriate');

capture sub {$context = wantarray};
ok(! defined($context), 'capture() calls subroutine in void context when appropriate');

# Test external program, see t/capture_exec.t for more
_reset;
capture sub {system($^X, '-V:archname')}, \$out;
like($out, "/$^O/", 'capture() caught stdout from external command');

# check we still get stdout/stderr if the code dies
eval {
    capture sub {print "."; print STDERR "5..4..3..2..1.."; die "self-terminating"}, \$out,\$err;
};
like($@, "/^self-terminating at " . quotemeta(__FILE__) . "/", 
    '$@ still available after capture');
ok($out eq '.' && $err eq '5..4..3..2..1..', 
    'capture() still populates output and error variables if the code dies');

# test fork()
sub forked_output {
    fork or do {
        print "forked";
        print STDERR "Child pid $$";
        exit;
    };
    select undef, undef, undef, 0.2;
}
capture \&forked_output, \$out, \$err;
ok($out eq 'forked' && $err =~ /^Child pid /, 'capture() traps fork() output');

# Test printing via C code
SKIP: {
    eval "require Inline::C";
    skip "Inline::C not available", 3 if $@;
    eval {
        my $c_code = do {local $/; <DATA>};
#        Inline->bind( 'C' => $c_code, FORCE_BUILD => 1, BUILD_NOISY => 1 );
        Inline->bind( 'C' => $c_code, FORCE_BUILD => 1);
    };
    skip "Inline->bind failed : $@", 3 if $@;
    ok(test_inline_c(), 'Inline->bind succeeded');

    _reset && capture sub { print_stdout("Hello World") }, \$out, \$err;
    is($out, 'Hello World', 'captured stdout from C function');

    _reset && capture sub { print_stderr("Testing stderr") }, \$out, \$err;
    is($err, 'Testing stderr', 'captured stderr from C function');
}


__DATA__
// A basic sub to test that the bind() succeeded
#include <stdio.h>
int test_inline_c () { return 42; }

// print to stdout
void print_stdout (const char *template, ... ) { 
    va_list ap;
    va_start( ap, template );
    vfprintf( stdout, template, ap );
    va_end( ap );
    fflush(stdout);
}
 
// print to stderr
// avoiding fprintf because of segfaults on MSWin32 with some versions of
// ActiveState and some combinations of MSVC compiler
void print_stderr (const char *template, ... ) { 
    va_list ap;
    va_start( ap, template );
    vfprintf( stderr, template, ap );
    va_end( ap );
    fflush(stderr);
}
