package Fey::Literal;

use strict;
use warnings;

our $VERSION = '0.33';

use Fey::FakeDBI;
use Fey::Types;
use Scalar::Util qw( blessed looks_like_number );
use overload ();

# This needs to come before we load subclasses or shit blows up
# because we end up with a metaclass object that is a
# Class::MOP::Class, not Moose::Meta::Class.
use Moose;
use MooseX::SemiAffordanceAccessor;
use MooseX::StrictConstructor;

sub new_from_scalar
{
    shift;
    my $val = shift;

    return Fey::Literal::Null->new()
        unless defined $val;

    # Freaking Perl overloading is so broken! An overloaded reference
    # will not pass the type constraints, so we need to manually
    # convert it to a non-ref.
    if ( blessed $val && overload::Overloaded( $val ) )
    {
        # The stringification method will be derived from the
        # numification method if needed. This might produce strange
        # results in the case of something that overloads both
        # operations, like a number class that returns either 2 or
        # "two", but in that case the author of the class made our
        # life impossible anyway ;)
        $val = $val . '';
    }

    return looks_like_number($val)
           ? Fey::Literal::Number->new($val)
           : Fey::Literal::String->new($val);
}

sub id
{
    return $_[0]->sql('Fey::FakeDBI');
}

no Moose;

__PACKAGE__->meta()->make_immutable();

# Need to be loaded after this package is made immutable.
require Fey::Literal::Function;
require Fey::Literal::Null;
require Fey::Literal::Number;
require Fey::Literal::String;
require Fey::Literal::Term;

1;

__END__

=head1 NAME

Fey::Literal - Represents a literal piece of a SQL statement

=head1 SYNOPSIS

  my $literal = Fey::Literal->new_from_scalar($string_or_number_or_undef);

=head1 DESCRIPTION

This class represents a literal piece of a SQL statement, such as a
string, number, or function.

It is the superclass for several more specific C<Fey::Literal>
subclasses, and also provides short

=head1 METHODS

This class provides the following methods:

=head2 Fey::Literal->new_from_scalar($scalar)

Given a string, number, or undef, this method returns a new object of
the appropriate subclass. This will be either a
C<Fey::Literal::String>, C<Fey::Literal::Number>, or
C<Fey::Literal::Null>.

=head2 $literal->set_alias_name($alias)

Use this to explicitly set a literal's alias name for use in SQL.  If you don't
set this, it will be autogenerated as needed.

=head2 $literal->id()

Returns a unique id for a literal object.

=head2 $literal->sql()

=head2 $literal->sql_with_alias()

=head2 $literal->sql_or_alias()

Returns the appropriate SQL snippet.

Calling C<< $literal->sql_with_alias() >> causes a unique alias for
the literal to be created.

=head1 AUTHOR

Dave Rolsky, <autarch@urth.org>

=head1 BUGS

See L<Fey> for details on how to report bugs.

=head1 COPYRIGHT & LICENSE

Copyright 2006-2009 Dave Rolsky, All Rights Reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut
