require 'test/unit'
require 'dhelp'
require 'fileutils'

class TC_DhelpDocumentPool < Test::Unit::TestCase
  TEST_DOC_DIR_DATABASE = 'test/dddb'
  TEST_INDEX_FILE       = 'test/index.swish++'

  def doc_base_document(path)
    Dhelp::DocBaseDocument.new("test/doc-base-pool/#{path}")
  end

  def doc_id_set_in_pool(pool)
    doc_id_set = Set.new
    pool.each do |doc|
      doc_id_set << doc.document
    end
    doc_id_set
  end

  def setup
    @pool = Dhelp::DhelpDocumentPool.new(:doc_base_dir => ['test/doc-base-pool'],
                                         :doc_dir_database => TEST_DOC_DIR_DATABASE,
                                         :index_file       => TEST_INDEX_FILE,
                                         :indexer_config_file => 'swish++.conf')
    @doc_base_id_set = Set.new(['docbook-xsl-doc-html',
                                'pica-manual',
                                'pica-manual-2'])
  end

  def test_each
    assert_equal @doc_base_id_set, doc_id_set_in_pool(@pool)
  end

  def test_deregistration
    deregistered_docs = ['pica-manual']
    deregistered_docs.each do |doc|
      @pool.deregister("test/doc-base-pool/#{doc}")
    end
    assert_equal @doc_base_id_set - deregistered_docs,
      doc_id_set_in_pool(@pool)
  end

  def test_registration
    index_file = 'test/share-doc/pica/manual.html/index.html'
    doc_id     = 'pica-manual'
    doc        = doc_base_document(doc_id)
    # Make sure we're expecting the correct directory
    assert_equal index_file,
      doc.formats.find {|f| f.format.downcase == 'html'}.index
    # Register document, see if the containing directory is added
    @pool.register(doc)
    ddd = Dhelp::DocDirDatabase.open(BDB::RDONLY, TEST_DOC_DIR_DATABASE)
    assert_equal doc_id, ddd.info_for_path(File.dirname(index_file)).first
    ddd.close
  end

  def test_section_tree
    expected_section_tree = {
      'Apps'  => {:documents   => [],
                  :subsections => {
                     'Text' => {:documents => [doc_base_document('docbook-xsl-doc-html')], :subsections => {}}}},
      'Admin' => {:documents   => [doc_base_document('pica-manual'),
                                   doc_base_document('pica-manual-2')],
                  :subsections => {}}}
    actual_section_tree   = @pool.section_tree
    # Can't compare the whole thing because the memory addresses for the
    # DocBaseDocument objects are different :-(
    assert_equal expected_section_tree.keys, actual_section_tree.keys
    assert_equal expected_section_tree['Apps'][:subsections].keys,
                 actual_section_tree['Apps'][:subsections].keys
    assert_equal expected_section_tree['Apps'][:subsections]['Text'][:documents].map {|d| d.document},
                 actual_section_tree['Apps'][:subsections]['Text'][:documents].map {|d| d.document}
    assert_equal expected_section_tree['Admin'][:documents].map {|d| d.document},
                 actual_section_tree['Admin'][:documents].map {|d| d.document}
  end

  def teardown
    @pool = nil
    @doc_base_id_set = nil
    FileUtils.rm_f TEST_DOC_DIR_DATABASE
    FileUtils.rm_f TEST_INDEX_FILE
  end
end
