
=head1 NAME

Archive::Any - Single interface to deal with zips and tarballs

=head1 SYNOPSIS

  use Archive::Any;

  my $archive = Archive::Any->new($archive_file);

  my @files = $archive->files;

  $archive->extract;

  my $type = $archive->type;

  $archive->is_impolite;
  $archive->is_naughty;

=head1 DESCRIPTION

This module is a single interface for manipulating different archive
formats.  Tarballs, zip files, etc...

Currently only tar (with or without gzip) and zip are supported.

Currently only supports unpacking.

=over 4

=item B<new>
`
  my $archive = Archive::Any->new($archive_file);
  my $archive = Archive::Any->new($archive_file, $type);

Creates an Archive::Any object representing $file, but don't do anything
with it yet.

$type is optional.  It lets you force the file type in-case
Archive::Any can't figure it out.  'tar' or 'zip' is currently
accepted.

=item B<type>

  my $type = $archive->type;

Returns the type of archive this is.  Currently 'zip' or 'tar'.

=item B<is_impolite>

  my $is_impolite = $archive->is_impolite;

 Checks to see if this archive is going to unpack into the current
 directory rather than create its own.

=item B<is_naughty>

  my $is_naughty = $archive->is_naughty;

Checks to see if this archive is going to unpack B<outside> the
current directory.

=back

=head1 AUTHOR

Michael G Schwern E<lt>schwern@pobox.comE<gt>

=cut

package Archive::Any;
use Carp::Always;

use strict;
use warnings;
use vars qw($VERSION @ISA);

$VERSION = 0.06;

use File::Spec::Functions qw(rel2abs splitpath splitdir);
use File::Type;
use MIME::Types;
use Module::Find;
use Data::Dumper;

# my $a = Archive::Any->new( '/tmp/file.zip', 'zip' );

#
# Pick a plugin.
#
sub new {
    my ( $proto, $file, $type ) = @_;
    return undef unless -f $file;

    my $available;

    my @plugins = findsubmod Archive::Any;
    foreach my $plugin ( @plugins ) {
        eval "require $plugin";
        next if $@;

        my $a = $plugin->can_handle();
        if ( ref($a) eq 'ARRAY' ) {
            foreach my $h ( @{$a} ) {
                next if exists( $available->{$h} );
                $available->{$h} = $plugin;
            }
        } else {
            $available->{$a} = $plugin;
        }
    }

    my $handler;

    if ( $type ) {
        my $mime_type = MIME::Types->new()->mimeTypeOf( $type );

        do {
            die( "No handler available for type $type" );
        } unless exists( $available->{$type} );

        $handler = $available->{$type};

    } else {
        my $mime_type = File::Type->new()->checktype_filename( $file );

        # MIME::Types has a funky interface.
        # it's not immediately apparent until you try it in the debugger.

        my $mo = new MIME::Types;
        my MIME::Types $lol_wtf = $mo->type( $mime_type );

		print Dumper( $lol_wtf->extensions() );
		my @x = $lol_wtf->extensions();
		print Dumper( @x );
		my $extension = $x[0];

        do {
            die( "Can't determine the file extension for mime type: $mime_type" );
        } unless $extension;

        do {
			warn( Dumper( $available ) );
            die( "No handler available for type extension '$extension'" );
        } unless exists( $available->{$extension} );

       $handler = $available->{$extension};
    }

    return bless {
                  file => $file,
                  handler => $handler,
              }, $proto;
}


=item B<extract>

  $archive->extract;
  $archive->extract($directory);

Extracts the files in the archive to the given $directory.  If no
$directory is given, it will go into the current working directory.

=cut

sub extract {
    my $self = shift;
    my $dir = shift;

    if ( defined( $self->{forced} ) ) {
        print( "Forced\n" );
    } else {
        print( "Not forced.\n" );
    }
    if ( exists( $self->{mime} ) ) {
        print( "Mime looks good.\n" );
    }

    my $plugin = $self->{available}->{$self->{mime}};

    defined( $dir ) ? return $plugin->extract( $self->{file}, $dir ) : $plugin->extract( $self->{file} );
}

=item B<files>

  my @file = $archive->files;

A list of files in the archive.

=cut

sub files {
    my( $self, $file ) = @_;
    return undef unless $self->{mime};
	return $self->{handler}->files( $self->{file} );
}

sub is_impolite {
    my ($self) = shift;

    my @files       = $self->files;
    my $first_file  = $files[0];
    my ($first_dir) = splitdir($first_file);

    return grep( !/^\Q$first_dir\E/, @files ) ? 1 : 0;
}

sub is_naughty {
    my ($self) = shift;
    return ( grep { m{^(?:/|(?:\./)*\.\./)} } $self->files ) ? 1 : 0;
}

sub handler {
    my $self = shift;
    return $self->{handler};
}

1;
