package WWW::TheEchoNest::Track;

# ABSTRACT: Wrapper for The Echo Nest API of music intelligence track calls

use Moose;

{
    $WWW::TheEchoNest::Track::VERSION = '1.0';
}

extends 'WWW::TheEchoNest';

sub profile {
    my ($self,%extra) = @_;

    return $self->request_with_minimum('track/profile',\%extra);
}

sub upload {
    my ($self,%extra) = @_;
    
    if ( !exists $extra{url} || !exists $extra{track} ) {
        $self->last_error("Can't track/upload wihtout a url or track");
    }
    
    
    my $uri;
    
    if ( exists $extra{url} ) {
        $uri = $extra{url};
    } elsif ( exists $extra{track} ) {
        $uri = $extra{track};
        delete $extra{track};
        $extra{url} = $uri;
    }
    
    if (!exists $extra{filetype}) {
        my $filetype = $self->get_file_extension($uri) || 'invalid';
        
        if ($filetype ne 'invalid' ) {
            $extra{filetype} = $filetype;
        } else {
            return "track/upload url or track provided is not supported"; 
        }
    }    
    
    if ($self->debug) {
        print "in track/upload working with a $extra{filetype} uri" , "\n";
    }
    
    if ( !exists $extra{force} ) {
        
        my $track_md5 = $self->get_resource_md5($uri); 
                  
        # warn "url: $extra{url} $track_md5";
        # this is weird, we have to call ourselves in a way...
        
        if (length($track_md5) == 32) {
            
            my $result = $self->request_with_minimum('track/profile',
                                        { md5 => $track_md5 }
                                        );
                
            if ( $self->get( 'status.message' ) eq 'Success' ) {
                # good news they have already seen it!
                return $result;
            }
        }
    }
    

    
    if (exists $extra{force}) {
        delete $extra{force};
    }
    
    return $self->send_post_request('track/upload',\%extra);
     
}

no Moose;
__PACKAGE__->meta->make_immutable;

1;

__END__

=pod

=head1 NAME

WWW::TheEchoNest::Track - Wrapper for The Echo Nest API of music intelligence track calls

=head1 VERSION

version 0.3

=head1 SYNOPSIS

 use WWW::TheEchoNest::Track;
 my $track = WWW::TheEchoNest::Track->new( api_key => $api_key );
 
 $track->profile(
    id     => 'TRTLKZV12E5AC92E11',
    bucket => 'audio_summary' 
 );

=head1 DESCRIPTION

Track API Methods

This call can be a bit confusing based on The Echo Nest API documentation and
the use case for each type of call may not be clear to people starting out so
I am providing a bit more documentation then on some of the other modules to hopefully
clarify how to use these calls.

A complete analysis of an audio recording, which is simply refered to as a "track"
in The Echo Nest API, can be provided by using the profile call on an existing piece of audio that
you already have an an md5 file *contents* checksum (don't worry too much about that, this has been provided
under the hood) for or new analysis can be provided by uploading
an audio file in one of the following formats;

wav mp3 au ogg m4a mp4

In an attempt to keep this as simply as possible in the context of this module
you can use either track or url in the same way and they should do what you expect.

The analysis may not be immediate so be sure and keep track of the md5 returned from
the request.

Possible analysis statuses are documented on The Echo Nest developer site.

You can get just an md5 of a file by using the get_resource_md5 method documented
in WWW::TheEchoNest

Another thing that might be slightly confusing is the difference with the audio_md5 and the
md5 itself.  The audio_md5 is *not* the md5 that profile expects. audio_md5 is generated by
the echonest and you are not able to reproduce it using this module.

The data returned by the profile method can make your applications much more robust and
allow you better tune the audio experience. For example as you monitor a listeners
choice in songs you can identity trends toward different attributes in the 'audio_summary'
section of the profile call response *if* you call it with the bucket audio_summary
parameter.

Reminder: the 'bucket' parameter can be either a single item or or multiple items
as an arrayref.  See WWW::TheEchoNest for more details.

=head1 NAME

WWW::TheEchoNest::Track

=head1 METHODS

=head2 profile

If you have a track id you can get track summary

 $track->profile(
    id     => 'TRTLKZV12E5AC92E11',
    bucket => 'audio_summary'
 );

If you have an md5 of an available or previously uploaded track

 my $track_md5 = '0a65922f961d9401dfbcc84c30e0094b';
 
 $track->profile(
    md5 => $track_md5 
 );

or

 my $track_id = 'TRTLKZV12E5AC92E11';
 
 $track->profile(
    id =>  $track_id
    bucket => [ 'audio_summary' ]
 );
 
 my $audio_summary = $track->get( 'track.audio_summary' );
 
 # $audio_summary will contain a hashref of track attributes if available
 
 print $audio_summary->{'energy'} , "\n";

See also the convience method documented in WWW::TheEchoNest for how to get
an md5 from your own audio.

=head2 upload

 $track->upload(
        url => 'http://www.example.com/test.mp3',
        filetype => 'mp3'
 );
 
 my $analysis_status = $track->get( 'track.status' );

This method attempts to avoid uploading tracks (file) that already have analysis
in The Echo Nest system.  It does this for local files and remote (http or ftp).

This method differs slightly from the API documentation in that it allows for a
force paramter to be passed if you want to skip the attempt to check for an existing
match.

=head1 AUTHOR

Aaron Johnson <aaronjjohnson@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2013 by Aaron Johnson.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
